/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.indexing.incremental.mail;

import java.util.Optional;

import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.indexing.IMailIndexService;
import net.bluemind.backend.mail.replica.indexing.RecordIndexActivator;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.indexing.incremental.AbstractTypeIndexerByUid;
import net.bluemind.indexing.incremental.AbstractTypeIndexerByUid.UidIndexWriter;
import net.bluemind.indexing.incremental.NoopIndexer;
import net.bluemind.indexing.incremental.TypeIndexerFactory;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;

public class FoldersIncrementalIndexFactory implements TypeIndexerFactory {

	@Override
	public String type() {
		return IMailReplicaUids.REPLICATED_MBOXES;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {
		IDbReplicatedMailboxes recApi = ctx.provider().instance(IDbByContainerReplicatedMailboxes.class, c.uid);
		IMailboxes mboxApi = ctx.provider().instance(IMailboxes.class, c.domainUid);
		ItemValue<Mailbox> mbox = mboxApi.getComplete(c.owner);
		Optional<IMailIndexService> optIdx = RecordIndexActivator.getIndexer();
		return optIdx.<TypeIndexer>map(midx -> new RecIndexer(recApi, new FolderIndexWriter(mbox, midx)))
				.orElseGet(NoopIndexer::new);

	}

	public static class FolderIndexWriter implements UidIndexWriter<MailboxFolder> {

		private IMailIndexService mailIndex;
		private ItemValue<Mailbox> mbox;

		public FolderIndexWriter(ItemValue<Mailbox> mbox, IMailIndexService midx) {
			this.mailIndex = midx;
			this.mbox = mbox;
		}

		@Override
		public void write(ItemValue<MailboxFolder> toIndex) {
			// only folders deletions have an impact on mail indices
		}

		@Override
		public void deleteByUid(String uid, String type) {
			mailIndex.deleteBox(mbox, uid);
		}

	}

	public static class RecIndexer extends AbstractTypeIndexerByUid<MailboxFolder> {

		public RecIndexer(IDbReplicatedMailboxes rec, UidIndexWriter<MailboxFolder> writer) {
			super(rec, rec, writer);
		}

	}

}
