package net.bluemind.central.reverse.proxy.model.impl;

import java.util.Objects;
import java.util.Optional;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.RateLimiter;

import io.vertx.core.Future;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.MessageProducer;
import io.vertx.core.json.JsonObject;
import net.bluemind.central.reverse.proxy.common.ProxyEventBusAddress;
import net.bluemind.central.reverse.proxy.model.RecordHandler;
import net.bluemind.central.reverse.proxy.model.client.PostfixMapsStoreClient;
import net.bluemind.central.reverse.proxy.model.client.ProxyInfoStoreClient;
import net.bluemind.central.reverse.proxy.model.common.DomainSettings;
import net.bluemind.central.reverse.proxy.model.common.InstallationInfo;
import net.bluemind.central.reverse.proxy.model.common.mapper.RecordKey;
import net.bluemind.central.reverse.proxy.model.common.mapper.RecordKeyMapper;
import net.bluemind.central.reverse.proxy.model.common.mapper.RecordValueMapper;

public class ByteArrayRecordHandler implements RecordHandler<byte[], byte[]> {
	private final Logger logger = LoggerFactory.getLogger(ByteArrayRecordHandler.class);

	private final Vertx vertx;
	private final ProxyInfoStoreClient proxyInfoStoreClient;
	private final PostfixMapsStoreClient postfixMapsStoreClient;
	private final RecordKeyMapper<byte[]> keyMapper;
	private final RecordValueMapper<byte[]> valueMapper;
	private final ProgressNotifier progress;

	public ByteArrayRecordHandler(Vertx vertx, ProxyInfoStoreClient proxyInfoStoreClient,
			PostfixMapsStoreClient postfixMapsStoreClient, RecordKeyMapper<byte[]> keyMapper,
			RecordValueMapper<byte[]> valueMapper) {
		this.vertx = vertx;
		this.proxyInfoStoreClient = proxyInfoStoreClient;
		this.postfixMapsStoreClient = postfixMapsStoreClient;
		this.keyMapper = keyMapper;
		this.valueMapper = valueMapper;
		this.progress = new ProgressNotifier(vertx);
	}

	public static class ProgressNotifier {
		private final RateLimiter notif = RateLimiter.create(0.1);
		private MessageProducer<String> publisher;

		public ProgressNotifier(Vertx v) {
			this.publisher = v.eventBus().publisher("model.records.progress");
		}

		public void notifyProgress() {
			if (notif.tryAcquire()) {
				publisher.write("REC");
			}
		}
	}

	@Override
	public void handle(ConsumerRecord<byte[], byte[]> rec) {
		progress.notifyProgress();
		keyMapper.map(rec.key()).flatMap(key -> safeStore(rec, key))
				.ifPresent(futureStoredKey -> futureStoredKey
						.onSuccess(storedKey -> logger.trace("[model] Stored {}:{}", storedKey.type(), storedKey))
						.onFailure(t -> logger.error("[model] Failed to store: {}", keyMapper.map(rec.key()), t)));
	}

	public Optional<Future<RecordKey>> safeStore(ConsumerRecord<byte[], byte[]> rec, RecordKey key) {
		try {
			return store(rec, key);
		} catch (Exception r) {
			return Optional.of(Future.failedFuture(r));
		}
	}

	private Optional<Future<RecordKey>> store(ConsumerRecord<byte[], byte[]> rec, RecordKey key) {
		if (rec.value() == null) {
			// CREATE or UPDATE operation waiting for compaction
			return Optional.empty();
		}

		switch (key.type()) {
		case "installation":
			return valueMapper.mapInstallation(key.uid(), rec.value())
					.map(installation -> Future.all(proxyInfoStoreClient.addInstallation(installation).map(oldIp -> {
						if (Objects.nonNull(oldIp)) {
							publishInstallationIpChange(installation, oldIp);
						}

						if (installation.hasCore) {
							publishCoreIp(installation.ip);
						}

						return key;
					}), postfixMapsStoreClient.addInstallation(installation)).map(v -> key));

		case "domains":
			byte[] val = rec.value();
			if ("net.bluemind.domain.api.Domain".equals(key.valueClass())) {
				return valueMapper.mapDomain(val)
						.map(domain -> Future
								.all(proxyInfoStoreClient.addDomain(domain), postfixMapsStoreClient.addDomain(domain))
								.map(v -> key));
			}

			if ("net.bluemind.domain.api.DomainSettings".equals(key.valueClass())) {
				Optional<DomainSettings> set = valueMapper.mapDomainSettings(val);
				return set
						.map(domainSettings -> postfixMapsStoreClient.addDomainSettings(domainSettings).map(v -> key));
			}

			return Optional.empty();

		case "dir":
			if (!"net.bluemind.directory.service.DirEntryAndValue".equals(key.valueClass())) {
				logger.debug("Unsupported dir entry value class {}", key.valueClass());
				return Optional.empty();
			}

			if (key.operation() != null && key.operation().equals("DELETE")) {
				return valueMapper.getValueUid(rec.value())
						.map(deletedUid -> postfixMapsStoreClient.removeDir(deletedUid).map(v -> key));
			}

			String domainUid = key.uid();
			return valueMapper.mapDir(domainUid, rec.value()).map(dir -> Future
					.all(proxyInfoStoreClient.addDir(dir), postfixMapsStoreClient.addDir(dir)).map(v -> key));

		case "memberships":
			return valueMapper.mapMemberShips(rec.value())
					.map(member -> postfixMapsStoreClient.manageMember(member).map(v -> key));

		default:
			logger.debug("[model] Skipping unknown type {}", key);
		}

		return Optional.empty();
	}

	private void publishCoreIp(String ip) {
		if (ip != null) {
			logger.info("[model] Announcing new core ip change to {}", ip);
			JsonObject jsonIp = new JsonObject().put("ip", ip);
			vertx.eventBus().publish(ProxyEventBusAddress.ADDRESS, jsonIp, ProxyEventBusAddress.CORE_IP_UPDATE);
		}
	}

	private void publishInstallationIpChange(InstallationInfo installation, String oldIp) {
		if (!installation.ip.equals(oldIp) && Objects.nonNull(oldIp)) {
			logger.info("[model] Announcing installation ip change for {}: {} -> {}", installation.dataLocationUid,
					oldIp, installation.ip);
			JsonObject ip = new JsonObject().put("ip", oldIp);
			vertx.eventBus().publish(ProxyEventBusAddress.ADDRESS, ip, ProxyEventBusAddress.INSTALLATION_IP_CHANGE);
		}
	}

}
