/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.delivery.smtp.ndr;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.lang.ref.Reference;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.ResourceBundle;

import org.apache.james.mime4j.dom.BinaryBody;
import org.apache.james.mime4j.dom.Header;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.Multipart;
import org.apache.james.mime4j.dom.TextBody;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.field.Fields;
import org.apache.james.mime4j.message.BasicBodyFactory;
import org.apache.james.mime4j.message.BodyPart;
import org.apache.james.mime4j.message.DefaultMessageBuilder;
import org.apache.james.mime4j.message.HeaderImpl;
import org.apache.james.mime4j.message.MessageImpl;
import org.apache.james.mime4j.message.MultipartImpl;
import org.apache.james.mime4j.stream.RawField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import freemarker.template.Configuration;
import freemarker.template.TemplateException;
import net.bluemind.common.freemarker.FreeMarkerMsg;
import net.bluemind.common.freemarker.MessagesResolver;
import net.bluemind.mime4j.common.Mime4JHelper;
import net.bluemind.mime4j.common.Mime4JHelper.SizedStream;

public class NonDeliveryReportMessage {
	private static final Logger logger = LoggerFactory.getLogger(NonDeliveryReportMessage.class);
	private static final String BASE_64 = "base64";

	private final List<FailedRecipient> failedRecipients;
	private final Message relatedMsg;
	private final Locale locale;
	private final SendmailResponse sendmailResponse;

	public NonDeliveryReportMessage(SendmailResponse response, Message relatedMsg, Locale locale) {
		this.failedRecipients = response.getFailedRecipientsInError();
		this.relatedMsg = relatedMsg;
		this.locale = locale;
		this.sendmailResponse = response;
	}

	public MessageImpl createNDRMessage() {
		MessagesResolver resolver = new MessagesResolver(
				ResourceBundle.getBundle("OSGI-INF/l10n/ndrTemplates", locale));

		MessageImpl message = new MessageImpl();
		message.setSubject(relatedMsg.getSubject());
		message.setSubject(resolver.translate("ndr.subject", new Object[] { message.getSubject() }));

		Multipart mp = new MultipartImpl("report");
		generateHtmlContent(resolver).ifPresentOrElse(c -> mp.addBodyPart(buildHtmlMsgBodyPart(c)),
				() -> logger.error("Cannot generate html content"));
		buildOriginalMessageAsAttachmentPart().ifPresentOrElse(mp::addBodyPart,
				() -> logger.error("Cannot attach EML"));
		buildDeliveryStatusPart().ifPresentOrElse(mp::addBodyPart,
				() -> logger.error("Cannot attach message/delivery-status"));

		message.setMultipart(mp, Map.of("report-type", "delivery-status"));
		return message;
	}

	private Optional<String> generateHtmlContent(MessagesResolver resolver) {
		HashMap<String, FailedRecipientBlock> blocks = new HashMap<>();
		failedRecipients.forEach(recipient -> {
			String key = String.valueOf(recipient.code) + "_" + recipient.smtpStatus.replace(".", "");
			if (blocks.containsKey(key)) {
				blocks.get(key).addRecipient(recipient);
			} else {
				ArrayList<FailedRecipient> recipients = new ArrayList<>();
				recipients.add(recipient);
				blocks.put(key, new FailedRecipientBlock(recipients, relatedMsg.getFrom().get(0), sendmailResponse));
			}
		});
		Map<String, Object> ftlDatas = new HashMap<>();
		ftlDatas.put("msg", new FreeMarkerMsg(resolver));

		ftlDatas.put("subject", relatedMsg.getSubject());
		ftlDatas.put("sendDate", relatedMsg.getDate() != null ? printDate(relatedMsg.getDate()) : "");

		ftlDatas.put("recipients", failedRecipients.stream().map(recipient -> recipient.email).toList());

		if (blocks.isEmpty()) {
			String key = String.valueOf(sendmailResponse.getCode()) + "_"
					+ sendmailResponse.getSmtpStatus().replace(".", "");
			blocks.put(key,
					new FailedRecipientBlock(Collections.emptyList(), relatedMsg.getFrom().get(0), sendmailResponse));
		}

		ftlDatas.put("recipientBlocks", blocks.values());

		return applyTemplate(ftlDatas);
	}

	private Optional<String> applyTemplate(Map<String, Object> ftlData) {
		try {
			StringWriter sw = new StringWriter();
			Configuration cfg = new Configuration(Configuration.DEFAULT_INCOMPATIBLE_IMPROVEMENTS);
			cfg.setClassForTemplateLoading(this.getClass(), "/templates");
			cfg.getTemplate("NonDeliveryReport.ftl", locale).process(ftlData, sw);
			sw.flush();
			return Optional.of(sw.toString());
		} catch (TemplateException | IOException e) {
			logger.error(e.getMessage());
			return Optional.empty();
		}
	}

	private static String printDate(Date date) {
		String dateStr = DateUtil.toDateHeader(date);
		dateStr = dateStr.replace("Date: ", "");
		return dateStr.substring(0, dateStr.length() - 7); // remove end ' +0000'
	}

	public record FailedRecipientBlock(List<FailedRecipient> recipients, Mailbox sender,
			SendmailResponse sendmailResponse) {

		protected void addRecipient(FailedRecipient recipient) {
			recipients.add(recipient);
		}

		public List<String> getEmails() {
			return recipients.stream().map(recipient -> recipient.email).toList();
		}

		public int getCode() {
			return !recipients.isEmpty() ? recipients.get(0).code : sendmailResponse.getCode();
		}

		public String getSmtpStatus() {
			return !recipients.isEmpty() ? recipients.get(0).smtpStatus : sendmailResponse.getSmtpStatus();
		}

		public String getFullCode() {
			return getCode() + " " + getSmtpStatus();
		}

		public String getErrorMsg() {
			return !recipients.isEmpty() ? recipients.get(0).message : sendmailResponse.getMessage();
		}

		public String getFrom() {
			return sender.getAddress();
		}

		public String getTransportAgent() {
			return sender.getDomain();
		}

		public String getI18n() {
			if (!SendmailHelper.isSuccessCode(getCode()) && recipients.isEmpty()) {
				return "norecipient";
			} else if (SendmailHelper.isSenderUnauthorized(getCode(), getSmtpStatus())) {
				return "550_rights";
			} else if (SendmailHelper.isTooBigFile(getCode(), getSmtpStatus())) {
				return "552_toobigfile";
			} else if (SendmailHelper.isNetworkError(getCode())) {
				return "400";
			} else {
				return String.valueOf(getCode());
			}
		}

	}

	private static BodyPart buildHtmlMsgBodyPart(String content) {
		TextBody body = new BasicBodyFactory().textBody(content, StandardCharsets.UTF_8);
		BodyPart bodyPart = new BodyPart();
		bodyPart.setBody(body);
		HeaderImpl h = new HeaderImpl();
		h.setField(Fields.contentType("text/html; charset=utf-8"));
		bodyPart.setHeader(h);
		bodyPart.setContentTransferEncoding(BASE_64);
		return bodyPart;
	}

	private Optional<BodyPart> buildOriginalMessageAsAttachmentPart() {
		InputStream originalMsg = sendmailResponse.getOriginalMsg();
		if (originalMsg == null) {
			return Optional.empty();
		}
		SizedStream sstream = null;
		try {
			InputStream data = null;
			if (SendmailHelper.isTooBigFile(sendmailResponse.getCode(), sendmailResponse.getSmtpStatus())) {
				Message message = Mime4JHelper.parse(originalMsg);
				message.removeBody();
				message.setBody(new BasicBodyFactory().textBody("", StandardCharsets.UTF_8));
				message.getHeader().removeFields("Content-Type");
				try {
					sstream = Mime4JHelper.asStream(message);
					data = sstream.input;
				} catch (IOException e) {
					logger.error(e.getMessage());
					return Optional.empty();
				}
			} else {
				data = originalMsg;
			}

			try {
				BodyPart bpa = new BodyPart();
				BinaryBody bb = new BasicBodyFactory().binaryBody(data);
				bpa.setBody(bb, "message/rfc822");
				bpa.setContentTransferEncoding(BASE_64);
				bpa.setFilename(Mime4JHelper.safeEncodeFilename(relatedMsg.getSubject() + ".eml"));
				return Optional.of(bpa);
			} catch (Exception e) {
				logger.error(e.getMessage());
				return Optional.empty();
			}
		} finally {
			Reference.reachabilityFence(sstream);
		}
	}

	private Optional<BodyPart> buildDeliveryStatusPart() {
		MessageImpl msgDeliveryStatusPart = deliveryStatusMsgImpl();
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		Mime4JHelper.serialize(msgDeliveryStatusPart, out);
		byte[] data = out.toByteArray();

		if (data == null) {
			return Optional.empty();
		}

		try {
			BodyPart bpa = new BodyPart();
			BinaryBody bb = new BasicBodyFactory().binaryBody(data);
			bpa.setBody(bb, "message/rfc822");

			Header mdnHeader = new DefaultMessageBuilder().newHeader();
			RawField contentDescHeader = new RawField("Content-Description", "Delivery report");
			mdnHeader.addField(contentDescHeader);
			mdnHeader.addField(Fields.contentType("message/delivery-status"));
			bpa.setHeader(mdnHeader);
			bpa.setContentTransferEncoding(BASE_64);
			bpa.setFilename(Mime4JHelper.safeEncodeFilename("details.txt"));
			return Optional.of(bpa);
		} catch (Exception e) {
			logger.error(e.getMessage());
			return Optional.empty();
		}
	}

	private MessageImpl deliveryStatusMsgImpl() {
		Header deliveryHeader = new HeaderImpl();
		deliveryHeader.addField(new RawField("MessageId", relatedMsg.getMessageId()));
		deliveryHeader.addField(
				new RawField("Reporting-MTA", String.format("dns; %s", relatedMsg.getFrom().get(0).getDomain())));
		deliveryHeader.addField(new RawField("X-Original-Message-ID", String.format("%s", relatedMsg.getMessageId())));
		if (sendmailResponse.isSenderDifferentFrom()) {
			deliveryHeader
					.addField(new RawField("X-Original-From", String.format("%s", sendmailResponse.getOriginalFrom())));
			deliveryHeader.addField(
					new RawField("X-Original-Sender", String.format("%s", sendmailResponse.getOriginalSender())));
		}

		MessageImpl deliveryMsg = new MessageImpl();
		deliveryMsg.setHeader(deliveryHeader);
		deliveryMsg.setDate(new Date());
		deliveryMsg.setSubject(relatedMsg.getSubject());

		StringBuilder mdnContent = new StringBuilder();
		for (FailedRecipient rcpt : failedRecipients) {
			mdnContent.append("\r\n");
			mdnContent.append(new RawField("Final-Recipient", String.format("rfc822; %s", rcpt.email))).append("\r\n");
			mdnContent.append(new RawField("Action", "failed")).append("\r\n");
			mdnContent.append(new RawField("Status", rcpt.smtpStatus)).append("\r\n");
			if (rcpt.email.contains("@")) {
				String serverDomain = rcpt.email.split("@")[1];
				mdnContent.append(new RawField("Remote-MTA", String.format("dns; %s", serverDomain))).append("\r\n");
			}
			mdnContent.append(new RawField("Diagnostic-Code",
					String.format("smtp; %d %s", rcpt.code, "\"" + rcpt.message.replace("<", "[").replace(">", "]"))))
					.append("\"\r\n");
		}

		deliveryMsg.setBody(new BasicBodyFactory().textBody(mdnContent.toString(), StandardCharsets.UTF_8));
		return deliveryMsg;
	}

}
