/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.backend.bm.mail;

import java.util.Arrays;
import java.util.Iterator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.VEvent;
import net.bluemind.calendar.api.VEventCounter;
import net.bluemind.calendar.api.VEventCounter.CounterOriginator;
import net.bluemind.calendar.api.VEventOccurrence;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.core.api.date.BmDateTimeWrapper;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.icalendar.api.ICalendarElement;
import net.bluemind.icalendar.api.ICalendarElement.Attendee;
import net.bluemind.imip.parser.IMIPInfos;
import net.bluemind.imip.parser.ITIPMethod;

public class MailImipManager {

	private static final Logger logger = LoggerFactory.getLogger(MailImipManager.class);

	private final BackendSession bs;
	private final IMIPInfos infos;
	private final ICalendar cs;

	public MailImipManager(BackendSession bs, IMIPInfos infos, ICalendar cs) {
		this.bs = bs;
		this.infos = infos;
		this.cs = cs;
	}

	void processImipInfos() {
		// BM-4930, ACMS-196, and many more...
		if (infos.method == ITIPMethod.REPLY || infos.method == ITIPMethod.CANCEL
				|| infos.method == ITIPMethod.COUNTER) {
			EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger,
					" **** Device {} sends IMIP email, method {}. user {}", bs.getDevId(), infos.method,
					bs.getLoginAtDomain());

			infos.iCalendarElements.forEach(element -> element.attendees.stream()
					.filter(attendee -> attendee.mailto != null && (attendee.mailto.equals(bs.getLoginAtDomain())
							|| attendee.mailto.equals(bs.getUser().getDefaultEmail())))
					.forEach(attendee -> updateEventCounterStatus(element, attendee)));

		} else {
			EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
					" **** Device {} tried to send an IMIP email, we prevented it. Method: {}, user: {}", bs.getDevId(),
					infos.method, bs.getUser().getLoginAtDomain());
		}
	}

	private void updateEventCounterStatus(ICalendarElement element, Attendee attendee) {
		ItemValue<VEventSeries> event = cs.getComplete(infos.uid);
		if (event != null) {
			element = checkForRecurrenceException(element, event);
			if (element instanceof VEventOccurrence uOccurr) {
				VEventOccurrence occu = event.value.occurrence(uOccurr.recurid);
				if (occu != null) {
					updateStatus(occu, attendee);
					updateCounter(attendee, event, occu, element);
					cs.update(infos.uid, event.value, true);
				} else {
					EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
							"did not found in {} occurrence with recurid {}", infos.uid, uOccurr.recurid);
				}
			} else {
				updateStatus(event.value.main, attendee);
				updateCounter(attendee, event, VEventOccurrence.fromEvent(event.value.main, null), element);
				cs.update(infos.uid, event.value, true);
			}
		} else {
			EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger, "did not found event with uid {}", infos.uid);
		}
	}

	private ICalendarElement checkForRecurrenceException(ICalendarElement element, ItemValue<VEventSeries> event) {
		if (event.value.main == null && event.value.occurrences.size() == 1) {
			return VEventOccurrence.fromEvent((VEvent) element, event.value.occurrences.get(0).recurid);
		}
		return element;
	}

	private void updateCounter(Attendee attendee, ItemValue<VEventSeries> event, VEventOccurrence occurrence,
			ICalendarElement element) {
		if (infos.method == ITIPMethod.COUNTER || attendee.counter != null) {
			VEventCounter counter = new VEventCounter();
			counter.originator = CounterOriginator.from(attendee.commonName, attendee.mailto);

			// fix dates
			VEvent vevent = (VEvent) element;
			if (attendee.counter != null) {
				long duration = BmDateTimeWrapper.toTimestamp(occurrence.dtend.iso8601, occurrence.dtend.timezone)
						- BmDateTimeWrapper.toTimestamp(occurrence.dtstart.iso8601, occurrence.dtstart.timezone);

				occurrence.dtstart = BmDateTimeWrapper.fromTimestamp(
						BmDateTimeWrapper.toTimestamp(attendee.counter.iso8601, attendee.counter.timezone));
				occurrence.dtend = BmDateTimeWrapper.fromTimestamp(
						BmDateTimeWrapper.toTimestamp(occurrence.dtstart.iso8601, occurrence.dtstart.timezone)
								+ duration);
			} else {
				occurrence.dtstart = vevent.dtstart;
				occurrence.dtend = vevent.dtend;
			}

			counter.counter = occurrence;

			event.value.counters = Arrays.asList(counter);
		}
	}

	private void updateStatus(VEvent event, Attendee attendee) {
		Iterator<Attendee> it = event.attendees.iterator();
		while (it.hasNext()) {
			Attendee a = it.next();
			if (attendee.mailto.equals(a.mailto)) {
				// Set new part status
				a.partStatus = attendee.partStatus;
			}
		}
	}
}
