/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;
import com.datastax.oss.driver.api.core.cql.ResultSet;

import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.repository.IChangelogStore;
import net.bluemind.core.container.repository.IWeightProvider;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.repository.sequences.ISequenceStore;
import net.bluemind.repository.sequences.Sequences;

public class CqlChangelogStore extends CqlAbstractStore implements IChangelogStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlChangelogStore.class);

	public enum ChangeType {
		CREATE(0),

		UPDATE(1),

		DELETE(2);

		private byte op;

		private ChangeType(int b) {
			this.op = (byte) b;
		}
	}

	private final Container container;
	private final int delTtlSeconds;
	private final CqlItemStore itemStore;
	private final ISequenceStore seqs;

	public CqlChangelogStore(Container container, CqlSession session, ISequenceStore seqs, long deletionsTtl,
			TimeUnit unit, SecurityContext ctx) {
		super(session);
		this.container = container;
		this.seqs = seqs;
		this.delTtlSeconds = (int) unit.toSeconds(deletionsTtl);
		this.itemStore = new CqlItemStore(container, session, seqs, ctx);
	}

	@Override
	public void itemCreated(LogEntry entry) throws SQLException {
		itemStored(entry, ChangeType.CREATE);
	}

	@Override
	public void itemUpdated(LogEntry entry) throws SQLException {
		itemStored(entry, ChangeType.UPDATE);
	}

	private void itemStored(LogEntry le, ChangeType type) {
		PreparedStatement prep = session.prepare("""
				UPDATE t_changeset
				SET
				version = ?,
				weight_seed = ?,
				date = toUnixTimestamp(now()),
				uid = ?
				WHERE
				container_id = ? AND item_id = ? AND type = ?
				""");

		ResultSet upsert = session
				.execute(prep.bind((int) le.version, le.weightSeed, le.itemUid, container.id, le.internalId, type.op));
		if (!upsert.wasApplied()) {
			logger.warn("{} was not applied.", le);
		}
	}

	@Override
	public void itemDeleted(LogEntry le) {

		String cleanupQ = """
				DELETE FROM t_changeset
				WHERE container_id = ? AND item_id = ? AND type = ?
				""";
		var res = batch(//
				b("""
						UPDATE t_changeset
						USING TTL ?
						SET
						version = ?,
						weight_seed = ?,
						date = toUnixTimestamp(now()),
						uid = ?
						WHERE
						container_id = ? AND item_id = ? AND type = 2
						""", //
						delTtlSeconds, (int) le.version, le.weightSeed, le.itemUid, container.id, le.internalId),

				b(cleanupQ, container.id, le.internalId, ChangeType.UPDATE.op),
				b(cleanupQ, container.id, le.internalId, ChangeType.CREATE.op));
		if (!res.wasApplied()) {
			logger.warn("Failed to delete {}", le.itemUid);
		}

	}

	@Override
	public ContainerChangeset<ItemIdentifier> fullChangesetById(IWeightProvider wp, long from, long to)
			throws SQLException {
		return changesetById0(wp, from, ItemFlagFilter.all(),
				r -> ItemIdentifier.of(r.uid, r.itemId, r.version, r.date));
	}

	@Override
	public void deleteLog() throws SQLException {
		var del = session.prepare("""
				DELETE FROM t_changeset WHERE container_id = ?
				""");
		var res = session.execute(del.bind(container.id));
		if (res.wasApplied()) {
			logger.info("changeset deleted for {}", container.uid);
		}
	}

	@Override
	public void allItemsDeleted(String subject, String origin) throws SQLException {
		PreparedStatement notDeleted = session.prepare("""
				SELECT item_id, uid, version, type
				FROM t_changeset
				WHERE container_id = ?
				ORDER BY item_id
				""");
		String sn = Sequences.itemVersions(container.uid);
		ResultSet res = session.execute(notDeleted.bind(container.id));

		res.map(r -> {
			if (r.getByte(3) == 2) {
				return null;
			}
			return LogEntry.create(seqs.nextVal(sn), r.getString(1), r.getLong(0), 0L);
		}).forEach(le -> {
			if (le != null) {
				itemDeleted(le);
			}
		});
	}

	private record IdRow(long itemId, byte type, int version, long seed, String uid, Date date)
			implements Comparable<IdRow> {

		@Override
		public int compareTo(IdRow o) {
			if (o.itemId == itemId) {
				return Integer.compare(version, o.version);
			} else {
				return Long.compare(itemId, o.itemId);
			}
		}

	}

	private static final EntityCreator<IdRow> IDRC = r -> new IdRow(r.getLong(0), r.getByte(1), r.getInt(2),
			r.getLong(3), r.getString(4), Date.from(r.getInstant(5)));

	private List<IdRow> allChanges() {
		return map("""
				SELECT
				item_id, type, version, weight_seed, uid, date
				FROM t_changeset
				WHERE container_id = ?
				""", IDRC, voidPop(), container.id);
	}

	private List<IdRow> sinceChanges(long from) {
		return map("""
				SELECT
				item_id, type, version, weight_seed, uid, date
				FROM t_changeset_by_version
				WHERE container_id = ?
				AND version >= ?
				""", IDRC, voidPop(), container.id, (int) from);
	}

	private <R> ContainerChangeset<R> changesetById0(IWeightProvider wp, long from, ItemFlagFilter filter,
			Function<IdRow, R> toPayload) {
		List<IdRow> matching = from > 0 ? sinceChanges(from) : allChanges();
		Collections.sort(matching);
		Set<Long> seenCreates = new LinkedHashSet<>();
		Set<Long> seenUpdates = new LinkedHashSet<>();
		List<Long> deletions = new ArrayList<>();
		ContainerChangeset<R> cs = new ContainerChangeset<>();
		cs.version = -1;
		Map<Long, IdRow> idxByItemId = matching.stream()
				.collect(Collectors.toConcurrentMap(r -> r.itemId, r -> r, (a, b) -> b));
		List<Long> itemIds = idxByItemId.keySet().stream().sorted().toList();

		// poor man's hash join on t_container item

		Map<Long, Item> withFlags = itemStore.getMultipleById(itemIds).stream()
				.collect(Collectors.toMap(i -> i.id, i -> i, (a, b) -> b));
		matching.forEach(r -> processMatching(filter, from, seenCreates, seenUpdates, deletions, cs, withFlags, r));

		cs.created = seenCreates.stream().sorted((id1, id2) -> {
			IdRow r1 = idxByItemId.get(id1);
			IdRow r2 = idxByItemId.get(id2);
			return Long.compare(wp.weight(r1.seed), wp.weight(r2.seed));
		}).map(idxByItemId::get).map(toPayload::apply).toList();

		cs.updated = seenUpdates.stream().sorted((id1, id2) -> {
			IdRow r1 = idxByItemId.get(id1);
			IdRow r2 = idxByItemId.get(id2);
			return Long.compare(wp.weight(r1.seed), wp.weight(r2.seed));
		}).map(idxByItemId::get).map(toPayload::apply).toList();

		if (from == 0) {
			cs.deleted = Collections.emptyList();
		} else {
			cs.deleted = deletions.stream().sorted((id1, id2) -> {
				IdRow r1 = idxByItemId.get(id1);
				IdRow r2 = idxByItemId.get(id2);
				return Long.compare(wp.weight(r1.seed), wp.weight(r2.seed));
			}).map(idxByItemId::get).map(toPayload::apply).toList();
		}

		return cs;
	}

	private <R> void processMatching(ItemFlagFilter filter, long from, Set<Long> seenCreates, Set<Long> seenUpdates,
			List<Long> deletions, ContainerChangeset<R> cs, Map<Long, Item> withFlags, IdRow r) {
		cs.version = r.version;

		if (filter.skipVersions.contains((long) r.version) || r.version == from) {
			return;
		}

		long itemId = r.itemId;
		Item it = withFlags.get(itemId);
		if (it != null && !it.match(filter)) {
			seenCreates.remove(itemId);
			seenUpdates.remove(itemId);
			deletions.add(itemId);
			return;
		}

		switch (r.type) {
		case 0:
			seenCreates.add(itemId);
			break;
		case 1:
			if (from == 0) {
				seenCreates.add(itemId);
			} else if (!seenCreates.contains(itemId)) {
				seenUpdates.add(itemId);
			}
			break;
		case 2:
			if (seenCreates.contains(itemId)) {
				seenCreates.remove(itemId);
			} else {
				seenUpdates.remove(itemId);
				deletions.add(itemId);
			}
			break;
		default:
			break;
		}
	}

	@Override
	public ContainerChangeset<String> changeset(IWeightProvider wp, long from, long to) throws SQLException {
		return changesetById0(wp, from, ItemFlagFilter.all(), r -> r.uid);
	}

	@Override
	public ContainerChangeset<Long> changesetById(IWeightProvider wp, long from, long to) throws SQLException {
		return changesetById0(wp, from, ItemFlagFilter.all(), r -> r.itemId);
	}

	@Override
	public ContainerChangeset<ItemVersion> changesetById(IWeightProvider wp, long from, long to, ItemFlagFilter filter)
			throws SQLException {
		return changesetById0(wp, from, ItemFlagFilter.all(), r -> new ItemVersion(r.itemId, r.version, r.date));
	}

}
