/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.directory.api;

import net.bluemind.core.api.BMApi;

/**
 * Base data structure of all directory related entities
 */
@BMApi(version = "3")
public class BaseDirEntry {

	public String displayName;
	/**
	 * Unique entry uid
	 */
	public String entryUid;
	/**
	 * The {@link DirEntry}'s {@link BaseDirEntry.AccountType} type
	 */
	public AccountType accountType;
	/**
	 * The {@link BaseDirEntry.Kind} of the {@link DirEntry}
	 */
	public Kind kind = Kind.DOMAIN;

	/**
	 * The type of an account.<br>
	 * FULL represents an entity with all available applications (messaging,
	 * contact, agenda etc.)<br>
	 * SIMPLE represents an entity with limited access (no sharing) to the messaging
	 * application FULL_AND_VISIO represents an entity having advanced features to
	 * the video conferencing functionalities
	 */
	@BMApi(version = "3")
	public enum AccountType {
		FULL, SIMPLE, FULL_AND_VISIO;
	}

	/**
	 * The kind of object the entry represents.
	 */
	@BMApi(version = "3")
	public enum Kind {
		USER(true, "mailbox", null),

		GROUP(true, "group", null),

		RESOURCE(true, "resource", null),

		MAILSHARE(true, "mailshare", null),

		SHARED_MAILBOX(true, "shared-mailbox", "USER"),

		CALENDAR, ADDRESSBOOK, DOMAIN, ORG_UNIT, EXTERNALUSER;

		private final boolean mbox;
		private final String path;
		private final String dataprotectPrefix;
		private final String dnKind;

		private Kind() {
			this(false, "", null);
		}

		private Kind(boolean mbox, String dataprotectPrefix, String dnKind) {
			this.mbox = mbox;
			this.dataprotectPrefix = dataprotectPrefix;
			this.path = name().toLowerCase() + "s";
			this.dnKind = dnKind == null ? name() : dnKind;
		}

		public boolean hasMailbox() {
			return mbox;
		}

		public String pathComponent() {
			return path;
		}

		public String getDataprotectPrefix() {
			return dataprotectPrefix;
		}

		public String dnKind() {
			return dnKind;
		}

	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((entryUid == null) ? 0 : entryUid.hashCode());
		result = prime * result + ((kind == null) ? 0 : kind.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		BaseDirEntry other = (BaseDirEntry) obj;
		if (entryUid == null) {
			if (other.entryUid != null)
				return false;
		} else if (!entryUid.equals(other.entryUid))
			return false;
		if (kind != other.kind)
			return false;
		return true;
	}

}
