/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.domain.cql.store;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Stream;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BoundStatement;
import com.datastax.oss.driver.api.core.cql.Row;

import net.bluemind.core.container.model.Item;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.repository.IDomainStore;

public class CqlDomainStore extends CqlAbstractStore implements IDomainStore {

	public CqlDomainStore(CqlSession s) {
		super(s);
	}

	@Override
	public void create(Item item, Domain value) throws SQLException {
		BoundStatement insertDom = b("""
				insert into t_domain
				(name, aliases, global, default_alias, properties, item_id)
				values
				(?, ?, ?, ?, ?, ?)
				""", value.name, value.aliases, value.global, value.defaultAlias, value.properties, item.id);
		BoundStatement insertName = b("insert into idx_domain_by_alias (alias, item_uid) values (?,?)", value.name,
				value.name);
		batch(Stream.concat(Stream.of(insertDom, insertName), value.aliases.stream()
				.map(alias -> b("insert into idx_domain_by_alias (alias, item_uid) values (?,?)", alias, value.name)))
				.toArray(BoundStatement[]::new));
	}

	@Override
	public void update(Item item, Domain value) throws SQLException {
		Domain existing = get(item);
		HashSet<String> staleAlias = new HashSet<>(existing.aliases);
		staleAlias.removeAll(value.aliases);
		pruneAliases(staleAlias);
		create(item, value);
	}

	private void pruneAliases(Collection<String> staleAlias) {
		if (!staleAlias.isEmpty()) {
			voidCql("DELETE FROM idx_domain_by_alias WHERE alias IN ?", new ArrayList<>(staleAlias));
		}
	}

	@Override
	public void delete(Item item) throws SQLException {
		Domain existing = get(item);
		HashSet<String> staleAlias = new HashSet<>(existing.aliases);
		pruneAliases(staleAlias);
		voidCql("DELETE FROM t_domain WHERE item_id=?", item.id);
	}

	private static final EntityPopulator<Domain> DOM_POP = (Row r, int index, Domain value) -> {
		value.name = r.getString(0);
		value.aliases = r.getSet(1, String.class);
		value.global = r.getBoolean(2);
		value.defaultAlias = r.getString(3);
		value.properties = r.getMap(4, String.class, String.class);
		return index;
	};

	@Override
	public Domain get(Item item) throws SQLException {
		return unique("SELECT name, aliases, global, default_alias, properties FROM t_domain WHERE item_id=?",
				r -> new Domain(), DOM_POP, item.id);
	}

	@Override
	public List<Domain> getMultiple(List<Item> items) throws SQLException {
		return map("SELECT name, aliases, global, default_alias, properties FROM t_domain WHERE item_id IN ?",
				r -> new Domain(), DOM_POP, items.stream().map(i -> i.id).toList());
	}

	@Override
	public boolean exists(Item item) throws SQLException {
		Domain d = get(item);
		return d != null;
	}

	@Override
	public String findByNameOrAliases(String name) throws SQLException {
		return unique("SELECT item_uid FROM idx_domain_by_alias WHERE alias=?", r -> r.getString(0), voidPop(), name);
	}

}
