/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.webmodule.server.project;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.FileLocator;
import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.webmodule.server.resources.IResourceProvider;

public abstract class AbstractProject implements IResourceProvider {
	private static final Logger logger = LoggerFactory.getLogger(AbstractProject.class);

	protected final Bundle bundle;

	protected final Set<Path> resources = new HashSet<>();

	protected Optional<Path> root = Optional.empty();

	private Path relativeRoot;

	public AbstractProject(Bundle bundle) {
		this.bundle = bundle;
	}

	private Optional<Path> loadBundleResourceIntoFileSystem(String path) {
		return Optional.ofNullable(path).flatMap(rootPath -> getFileInBundle(rootPath));
	}

	private List<Path> findFilesInBundle(String pattern, String parent, boolean recurse) {
		List<Path> files = new ArrayList<>();
		Enumeration<URL> entries = bundle.findEntries(parent, pattern, true);
		while (entries.hasMoreElements()) {
			bundleURLToPath(entries.nextElement()).ifPresent(path -> files.add(path));
		}
		return files;
	}

	private Optional<Path> bundleURLToPath(URL url) {
		return Optional.ofNullable(url).map(value -> {
			try {
				return Paths.get(FileLocator.toFileURL(value).getFile());
			} catch (IOException e) {
				logger.warn("error during loading resource of {} ", getBundleName(), e);
				return null;
			}
		});
	}

	private String getBundlePath(Path path) {
		if (root.isPresent()) {
			return relativeRoot.resolve(root.get().relativize(path)).toString();
		} else {
			return path.toString();
		}
	}

	protected void setRootDir(String pathInBundle) {
		root = loadBundleResourceIntoFileSystem(pathInBundle);
		if (root.isEmpty()) {
			logger.warn("{} is specified as a public-resource provider rootPath ({}) is not found", getBundleName(),
					pathInBundle);
		} else {
			relativeRoot = Paths.get(pathInBundle);
		}
	}

	protected Optional<Path> getFileInBundle(String path) {
		return bundleURLToPath(bundle.getResource(path));
	}

	protected List<Path> findFilesInBundle(String pattern, Path parent) {
		return findFilesInBundle(pattern, getBundlePath(parent), true);
	};

	protected Path resolve(String path) {
		if (root.isPresent()) {
			return root.get().resolve(path);
		} else {
			return Paths.get(path);
		}
	}

	protected String getPublicPath(Path path) {
		if (root.isPresent()) {
			return root.get().relativize(path).toString();
		} else {
			return path.toString();
		}
	}

	protected String getAbsolutePath(Path path) {
		return path.toString();
	}

	protected void load(String relativePath) {
		load(resolve(relativePath));
	}

	protected void load(Path file) {
		if (root.isPresent() && !resources.contains(file)) {
			if (Files.exists(file)) {
				logger.debug("Load resource for bundle {} => {}", getBundleName(), file.toString());
				resources.add(file);
			} else {
				logger.debug("Ressource {} does not exist in  {}", file.toString(), getBundleName());

			}
		}
	}

	public Bundle getBundle() {
		return bundle;
	}

	public String getBundleName() {
		return bundle.getSymbolicName();
	}

	public Optional<Path> getResource(String relativePath) {
		Path resource = resolve(relativePath);
		if (resources.contains(resource)) {
			return Optional.of(resource);
		}
		logger.debug("{} not found into {} : {} ", relativePath, getBundleName());
		return Optional.empty();
	}

	public boolean hasResource(String relativePath) {
		return resources.contains(resolve(relativePath));
	}

	@Override
	public Collection<String> getResources() {
		return resources.stream().map(resource -> getPublicPath(resource)).collect(Collectors.toList());
	}
}
