/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.webmodule.server.handlers.internal;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HexFormat;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

import org.osgi.framework.Bundle;

/**
 * ETag generator service with persistent cache during application lifetime
 */
public class ETagGenerator {

	private static final ETagGenerator INSTANCE = new ETagGenerator();
	private static final int BUFFER_SIZE = 16 * 1024; // 16 KB buffer
	private static final HexFormat HEX_FORMAT = HexFormat.of();

	private final Map<String, String> etagCache = new ConcurrentHashMap<>();

	private ETagGenerator() {
	}

	public static ETagGenerator instance() {
		return INSTANCE;
	}

	/**
	 * Retrieves the ETag for a given file (with caching)
	 * 
	 * @param filePath   The file path
	 * @param bundleName The bundle name associated with the file
	 * @return The calculated or cached ETag
	 */
	public String getETag(Optional<Path> filePath, Bundle bundle) {
		if (filePath.isPresent()) {
			String cacheKey = buildCacheKey(filePath.get(), bundle.getSymbolicName());
			return etagCache.computeIfAbsent(cacheKey, k -> computeETag(filePath.get(), bundle));
		} else {
			return fallback(bundle);
		}
	}

	/**
	 * Builds a unique cache key based on path and bundle
	 */
	private String buildCacheKey(Path filePath, String bundleName) {
		return bundleName + ":" + filePath.toString();
	}

	/**
	 * Computes the ETag based on MD5 hash of content and bundleName
	 */
	private String computeETag(Path filePath, Bundle bundle) {
		try {
			// MD5 hash of file content
			MessageDigest md = MessageDigest.getInstance("MD5");
			try (InputStream is = Files.newInputStream(filePath);
					DigestInputStream dis = new DigestInputStream(is, md)) {

				byte[] buffer = new byte[BUFFER_SIZE];
				while (dis.read(buffer) != -1) {
				}
			}
			byte[] digest = md.digest();

			String contentHash = HEX_FORMAT.formatHex(digest);
			int bundleHash = bundle.getSymbolicName().hashCode();

			// Format: W/"bundleHash-contentHash"
			return "W/\"" + bundleHash + "-" + contentHash + "\"";

		} catch (IOException | NoSuchAlgorithmException e) {
			return fallback(bundle);
		}
	}

	private String fallback(Bundle bundle) {
		return "W/" + bundle.getSymbolicName().hashCode() + "-" + bundle.getVersion();

	}


}