/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keydb.sessiondata.cache;

import java.util.concurrent.TimeUnit;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.keydb.sessiondata.SessionData;

public class SessionDataCacheFactory {

	public static class Registration implements ICacheRegistration {
		private static final Cache<String, SessionData> cache = Caffeine.newBuilder()
				.expireAfterWrite(10, TimeUnit.SECONDS).recordStats().initialCapacity(4096).softValues().build();

		@Override
		public void registerCaches(CacheRegistry cr) {
			if (cache != null) {
				cr.register(SessionDataCacheFactory.class, cache);
			}
		}
	}

	public interface SessionDataCache {
		public default SessionData getIfPresent(String sid) {
			return null;
		}

		public default void put(String sid, SessionData sessionData) {
		}

		public default void invalidate(String authKey) {
		}
	}

	public class NoCache implements SessionDataCache {
	}

	public class WithCache implements SessionDataCache {
		private final Cache<String, SessionData> cache;

		public WithCache(Cache<String, SessionData> cache) {
			this.cache = cache;
		}

		@Override
		public SessionData getIfPresent(String sid) {
			return cache.getIfPresent(sid);
		}

		@Override
		public void put(String sid, SessionData sessionData) {
			cache.put(sid, sessionData);
		}

		@Override
		public void invalidate(String sid) {
			cache.invalidate(sid);
		}
	}

	public SessionDataCache get() {
		// Use local cache only from bm-webserver
		if (Registration.cache != null) {
			return new WithCache(Registration.cache);
		}

		return new NoCache();
	}
}
