/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.dataprotect.resource.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.RestoreRestorableItem;
import net.bluemind.dataprotect.directory.DirectoryBackupRepository;
import net.bluemind.dataprotect.resource.impl.pg.RestoreResourceTaskPg;

public class RestoreResourceTask extends BlockingServerTask implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreResourceTask.class);

	private final ResourceRestoreWorker resourceWorker;
	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;

	public RestoreResourceTask(DataProtectGeneration backup, Restorable item) {
		this.backup = backup;
		this.restorableItem = new RestoreRestorableItem(item);
		this.resourceWorker = new ResourceRestoreWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.RESOURCE.name()));
		DirectoryBackupRepository repository = DirectoryBackupRepository.getRepository(backup);
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreResourceTaskPg(backup, restorableItem, resourceWorker).run();
		}
	}

	protected void restore(DirectoryBackupRepository repository) {
		try {
			restorableItem.monitorLog("Restoring resource " + restorableItem.entryUid());
			repository.getRestorableResource(restorableItem.domain(), restorableItem.entryUid()).ifPresent(rn -> {
				try {
					resourceWorker.createOrUpdateLiveEntry(rn.item);
					resourceWorker.restoreEntryDependantContainers();
				} catch (InterruptedException e) {
					logger.error("Error while restoring resource, when trying to delete first existing resource", e);
					restorableItem.errors()
							.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.RESOURCE));
				}
			});

		} catch (Exception e) {
			logger.error("Error while restoring resource", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.RESOURCE));
		} finally {
			restorableItem.endTask();
		}
	}

}
