/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.resource.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.api.TaskStatus;
import net.bluemind.core.task.service.TaskUtils;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.RestoreDefinition;
import net.bluemind.dataprotect.calendar.RestoreUserCalendars;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.directory.RestoreDirWorker;
import net.bluemind.resource.api.IResources;
import net.bluemind.resource.api.ResourceDescriptor;

public class ResourceRestoreWorker extends RestoreDirWorker<ResourceDescriptor> {
	private static final Logger logger = LoggerFactory.getLogger(ResourceRestoreWorker.class);

	public ResourceRestoreWorker(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem) {
		super(backup, restorableItem);
	}

	@Override
	public ItemValue<ResourceDescriptor> createOrUpdateLiveEntry(ItemValue<ResourceDescriptor> backupResource)
			throws InterruptedException {

		IResources resourceServiceLive = live.instance(IResources.class, restorableItem.domain());

		ItemValue<ResourceDescriptor> liveResource = resourceServiceLive.getComplete(restorableItem.entryUid());
		if (liveResource != null) {
			resourceServiceLive.update(liveResource.uid, backupResource.value);
			restorableItem.setLiveEntryUid(liveResource.uid);
			return liveResource;
		}

		liveResource = resourceServiceLive.byEmail(backupResource.value.defaultEmail().address);
		if (liveResource != null) {
			TaskRef deleteTask = resourceServiceLive.delete(liveResource.uid);
			TaskStatus taskDeleteStatus = TaskUtils.waitForInterruptible(live, deleteTask);
			if (taskDeleteStatus.state.succeed) {
				restorableItem.monitorLog("Delete resource '{}' succeed {}", backupResource.value.label,
						taskDeleteStatus.result);
			}
		}

		resourceServiceLive.create(restorableItem.entryUid(), backupResource.value);
		liveResource = resourceServiceLive.getComplete(restorableItem.entryUid());
		restorableItem.setLiveEntryUid(liveResource.uid);
		return liveResource;
	}

	@Override
	public void restoreEntryDependantContainers() {
		logger.info("Restoring resource calendars {}:{}", restorableItem.entryUid(), restorableItem.displayName());
		RestoreDefinition restoreCalendars = new RestoreDefinition(RestoreUserCalendars.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreCalendars);
	}
}
