/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.authentication.mgmt.service;

import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.authentication.mgmt.api.ISessionsMgmt;
import net.bluemind.authentication.mgmt.api.SessionEntry;
import net.bluemind.authentication.mgmt.api.SessionUpdate;
import net.bluemind.authentication.service.logout.LogoutAction;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.sessions.Sessions;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.role.api.BasicRoles;

public class SessionsMgmt implements ISessionsMgmt {
	private static final Logger logger = LoggerFactory.getLogger(SessionsMgmt.class);

	private BmContext context;

	public SessionsMgmt(BmContext context) {
		this.context = context;
	}

	@Override
	public void logoutUser(String uid) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_SYSTEM_MANAGER);

		List<SecurityContext> userSessions = Sessions.get().asMap().values().stream()
				.filter(sc -> sc.getSubject().equals(uid)).toList();

		if (userSessions.isEmpty()) {
			return;
		}

		userSessions.forEach(this::invalidateSession);
	}

	private void invalidateSession(SecurityContext sc) {
		if (sc.getSessionId() == null) {
			return;
		}

		if (logger.isDebugEnabled()) {
			logger.debug("logout user {} session {}", sc.getSubject(), sc.getSessionId());
		}
		LogoutAction.full(sc.getSessionId()).logout();
	}

	@Override
	public List<SessionEntry> list(String domainUid) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_SYSTEM_MANAGER);
		return Sessions.get().asMap().values().stream()
				.filter(v -> domainUid != null ? v.getContainerUid().equals(domainUid) : true)
				.map(sc -> SessionEntry.build(sc.getCreated(),
						Optional.ofNullable(context.provider().instance(IDirectory.class, sc.getContainerUid())
								.findByEntryUid(sc.getSubject())).map(de -> de.email).orElse(null),
						sc.getContainerUid(), sc.getSubject(), sc.getOrigin(), sc.getRemoteAddresses()))
				.toList();
	}

	@Override
	public void updateCurrent(SessionUpdate ud) {
		RBACManager.forContext(context).checkNotAnoynmous();
		String sid = context.getSecurityContext().getSessionId();
		if (!context.getSecurityContext().fromGlobalVirt() && sid != null) {
			SecurityContext current = Sessions.get().getIfPresent(sid);
			Sessions.get().put(sid, current.from(ud.remoteIps));
			logger.debug("Remote IPs updated to {}", ud.remoteIps);
		}
	}
}
