/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.backup.continuous.state;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Date;
import java.util.concurrent.CompletableFuture;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.backup.continuous.DefaultBackupStore;
import net.bluemind.core.backup.continuous.api.InstallationWriteLeader;
import net.bluemind.core.backup.continuous.dto.Seppuku;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.stateobserver.IStateListener;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class LeaderStateListener implements IStateListener {

	private static final Logger logger = LoggerFactory.getLogger(LeaderStateListener.class);
	private SystemState cur;

	@Override
	public void stateChanged(SystemState newState) {
		if (newState == cur) {
			return;
		}
		if (newState == SystemState.CORE_STATE_DEMOTED) {
			demote();
		} else if (cur == SystemState.CORE_STATE_CLONING && newState == SystemState.CORE_STATE_RUNNING) {
			DefaultBackupStore.store().resume();
		}
		cur = newState;
	}

	private void demote() {
		InstallationWriteLeader leadership = DefaultBackupStore.store().leadership();
		ServerSideServiceProvider prov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);

		preventCoreStart();
		if (leadership.isLeader()) {
			writeByeMessageToKafka(prov).thenAccept(v -> leadership.releaseLeadership()).exceptionally(t -> {
				logger.error(t.getMessage(), t);
				return null;
			}).join();
		} else {
			logger.warn("{} says we are not leaders", leadership);
		}
	}

	private void preventCoreStart() {
		try {
			Path path = Paths.get("/etc/bm/bm-core.disabled");
			Files.createFile(path);
			logger.info("Wrote ({}) /etc/bm/bm-core.disabled to prevent further starts.", path);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
	}

	private CompletableFuture<Void> writeByeMessageToKafka(ServerSideServiceProvider prov) {
		IInCoreDomains domApi = prov.instance(IInCoreDomains.class);
		ItemValue<Domain> userDom = domApi.allUnfiltered().stream().filter(d -> !d.value.global).findFirst()
				.orElse(null);
		if (userDom == null) {
			logger.error("We did not find a single domain....");
			return CompletableFuture.failedFuture(ServerFault.notFound("no domain"));
		}
		logger.info("seppuku time, write Bye message for {}", userDom.uid);
		IContainers contApi = prov.instance(IContainers.class);
		BaseContainerDescriptor dirDesc = contApi.getLight(userDom.uid);
		Seppuku sep = new Seppuku();
		sep.byeTime = new Date();
		ItemValue<Seppuku> bye = ItemValue.create("seppuku", sep);
		bye.internalId = bye.uid.hashCode();
		bye.created = sep.byeTime;
		ITxOutbox txOutbox = prov.instance(ITxOutbox.class, dirDesc.domainUid, dirDesc.owner, dirDesc.type, dirDesc.uid,
				DataLocation.directory().serverUid());
		long suicideMsg = txOutbox.forKafka(bye, null, false);
		CompletableFuture<Void> v = new CompletableFuture<>();
		Thread.ofVirtual().name("seppuku-flusher").start(() -> {
			logger.info("Waiting for flush of outbox seq {}", suicideMsg);
			do {
				try {
					Thread.sleep(10);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
			} while (txOutbox.lastFlushedSeq() < suicideMsg);
			v.complete(null);
		});
		return v;
	}

}
