package net.bluemind.delivery.rules;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.util.concurrent.MoreExecutors;

import io.lettuce.core.SetArgs;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;
import io.lettuce.core.codec.ByteArrayCodec;
import net.bluemind.keydb.common.ClientProvider;

public class MailboxVacationSendersCache {

	public static class Factory {
		private static Map<String, Factory> INSTANCES = new HashMap<>();

		public static Factory build(String instanceName) {
			return INSTANCES.computeIfAbsent(instanceName,
					name -> new Factory(name, ClientProvider.newClient().connect(ByteArrayCodec.INSTANCE)));
		}

		public static Factory build() {
			return build("DEFAULT");
		}

		private final Cache<String, MailboxVacationSendersCache> mailboxesCache;
		private final StatefulRedisConnection<byte[], byte[]> redisConnection;
		private String name;

		private Factory(String name, StatefulRedisConnection<byte[], byte[]> redisConnection) {
			this.redisConnection = redisConnection;
			this.name = name;
			this.mailboxesCache = Caffeine.newBuilder() //
					.executor(MoreExecutors.directExecutor()) //
					.build();
		}

		public MailboxVacationSendersCache get(String mailboxUid) {
			return mailboxesCache.get(mailboxUid,
					key -> new MailboxVacationSendersCache(redisConnection, mailboxUid, name));
		}

		public void clear(String mailboxUid) {
			mailboxesCache.invalidate(mailboxUid);
		}
	}

	private static final byte[] CONST_VALUE = new byte[] { 0x01 };

	private final Duration expirationPeriod;
	private final String mailBoxPrefix;

	private final RedisCommands<byte[], byte[]> commands;

	private MailboxVacationSendersCache(StatefulRedisConnection<byte[], byte[]> redisConnection, String mailboxUid,
			String name, long ttl, TimeUnit ttlUnit) {
		this.expirationPeriod = Duration.ofMillis(ttlUnit.toMillis(ttl));
		this.mailBoxPrefix = new StringBuilder("vacation_reply_").append(mailboxUid).append("_").append(name)
				.toString();
		this.commands = redisConnection.sync();
	}

	private MailboxVacationSendersCache(StatefulRedisConnection<byte[], byte[]> redisConnection, String mailboxUid,
			String name) {
		this(redisConnection, mailboxUid, name, 3, TimeUnit.DAYS);
	}

	public <T> T ifMissingDoGetOrElseGet(String senderEmail, Supplier<T> doGet, Supplier<T> orElseGet) {
		if (senderEmail == null || contains(senderEmail)) {
			return orElseGet.get();
		}
		T result = doGet.get();
		put(senderEmail);
		return result;
	}

	public byte[] getKey(String senderMail) {
		return new StringBuilder(mailBoxPrefix).append("_").append(senderMail).toString().getBytes();
	}

	public boolean contains(String senderEmail) {
		return commands.exists(getKey(senderEmail)) > 0;
	}

	private void put(String senderEmail) {
		commands.set(getKey(senderEmail), CONST_VALUE, SetArgs.Builder.ex(expirationPeriod));
	}

}
