package net.bluemind.cql.sequences.zk;

import java.io.File;
import java.io.InputStream;
import java.nio.file.Files;
import java.time.Duration;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.IntStream;
import java.util.stream.LongStream;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.atomic.DistributedAtomicLong;
import org.apache.curator.retry.BoundedExponentialBackoffRetry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

import net.bluemind.config.InstallationId;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.rest.BmContext;
import net.bluemind.cql.persistence.CqlRepositoryFactory;
import net.bluemind.repository.provider.IStandaloneFactory;
import net.bluemind.repository.sequences.IItemSequencesStore;
import net.bluemind.repository.sequences.ISequenceStore;
import net.bluemind.repository.sequences.SequencePersistenceException;
import net.bluemind.repository.sequences.Sequences;

public class ZkSequenceStore implements ISequenceStore, IItemSequencesStore {

	private static final Logger logger = LoggerFactory.getLogger(ZkSequenceStore.class);

	private static final Set<StoreCapability> CAPS = EnumSet.of(StoreCapability.DURABLE,
			StoreCapability.PARTITION_TOLERANT, StoreCapability.DISTRIBUTED);

	private static Config loadConfig() {
		Config conf = ConfigFactory.load(ZkSequenceStore.class.getClassLoader(), "resources/cql-persistence.conf");
		try {
			File local = new File("/etc/bm/cql-persistence.conf"); // NOSONAR
			if (local.exists()) {
				Config parsed = ConfigFactory.parseFile(local);
				conf = parsed.withFallback(conf);
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

		Config systemPropertyConfig = ConfigFactory.defaultApplication();
		return systemPropertyConfig.withFallback(conf);
	}

	private static final Config conf = loadConfig();
	private static final String ROOT_NODE = "/bluemind/v5/" + InstallationId.getIdentifier().replace("bluemind-", "")
			+ "/sequences";
	private static final String ROOT = ROOT_NODE + "/";
	private static final ConcurrentHashMap<String, CuratorFramework> con = new ConcurrentHashMap<>();
	private static final Cache<String, WithPrefetch> locks = Caffeine.newBuilder()
			.expireAfterAccess(Duration.ofMinutes(10)).build();

	// This is set by InstallFromBackup/RestoreContainerSequence, it's a "global"
	// minimum value to use for all sequences
	private static final AtomicLong MINIMUM_SEQUENCE = new AtomicLong(0L);

	private record WithPrefetch(ConcurrentLinkedQueue<Long> prefetch, DistributedAtomicLong seq, AtomicLong lastVal) {

		private static final int PREFETCH_SIZE = conf.getInt("persistence.zk.prefetch");

		public long next() {
			try {
				synchronized (prefetch) {
					if (prefetch.isEmpty()) {
						prefetchUnlocked();
					}
					long ret = prefetch.poll().longValue();
					if (ret > MINIMUM_SEQUENCE.get()) {
						MINIMUM_SEQUENCE.set(ret);
					}
					lastVal.set(ret);
					return ret;
				}
			} catch (Exception e) {
				throw new SequencePersistenceException(e);
			}
		}

		private void prefetchUnlocked() throws Exception {
			long minimumValue = MINIMUM_SEQUENCE.get();
			long fresh = seq.add((long) PREFETCH_SIZE).postValue();
			if (fresh < minimumValue + PREFETCH_SIZE) {
				fresh = minimumValue + PREFETCH_SIZE;
				seq.forceSet(fresh);
			}
			LongStream.rangeClosed(fresh - PREFETCH_SIZE + 1, fresh).forEach(prefetch::add);
		}

		public long nextN(int count) {
			try {
				synchronized (prefetch) {
					while (prefetch.size() < count) {
						prefetchUnlocked();
					}
					long[] seqs = IntStream.range(0, count).mapToLong(i -> prefetch.poll()).toArray();
					long last = seqs[seqs.length - 1];
					if (last > MINIMUM_SEQUENCE.get()) {
						MINIMUM_SEQUENCE.set(last);
					}
					long rangeStart = seqs[0];
					lastVal.set(last);
					return rangeStart;
				}
			} catch (Exception e) {
				throw new SequencePersistenceException(e);
			}
		}

		public long cur() {
			if (MINIMUM_SEQUENCE.get() > 0) {
				return MINIMUM_SEQUENCE.get();
			}
			return lastVal.get();
		}
	}

	private CuratorFramework curatorFramework(Config c) {
		return con.computeIfAbsent("curator", k -> {
			String zkBoot = zkBootstrap(c);

			BoundedExponentialBackoffRetry rtp = new BoundedExponentialBackoffRetry(100, 10000, 15);
			CuratorFramework curator = CuratorFrameworkFactory.newClient(zkBoot, rtp);
			curator.start();
			try {
				curator.blockUntilConnected();
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				throw new SequencePersistenceException(e);
			}
			return curator;
		});
	}

	@Override
	public long nextVal(String sequenceName) {
		return distributedLong(sequenceName).next();
	}

	@Override
	public long nextVals(String sequenceName, int count) {
		WithPrefetch seq = distributedLong(sequenceName);
		return seq.nextN(count);
	}

	@Override
	public long curVal(String sequenceName) {
		return distributedLong(sequenceName).cur();
	}

	// Used by setup clone to bump sequences
	@Override
	public void setMinimumValue(long minimumSequence) {
		MINIMUM_SEQUENCE.set(minimumSequence);
	}

	private static long safeCurVal(DistributedAtomicLong seq) {
		try {
			return seq.get().postValue();
		} catch (Exception e) {
			throw new SequencePersistenceException(e);
		}
	}

	/**
	 * @param sequenceName eg. {@link Sequences#itemVersions(String)}
	 * @return
	 */
	private WithPrefetch distributedLong(String sequenceName) {
		return locks.get(sequenceName, sn -> {
			BoundedExponentialBackoffRetry rtp = new BoundedExponentialBackoffRetry(100, 10000, 15);
			DistributedAtomicLong seq = new DistributedAtomicLong(curatorFramework(conf), ROOT + sn, rtp);
			return new WithPrefetch(new ConcurrentLinkedQueue<>(), seq, new AtomicLong(safeCurVal(seq)));
		});
	}

	private static String zkBootstrap(Config config) {
		String zkBootstrap = config != null && config.hasPath("persistence.zk.servers")
				? config.getString("persistence.zk.servers")
				: null;
		if (zkBootstrap == null) {
			File local = new File("/etc/bm/kafka.properties");
			logger.warn("Loading from legacy {}", local.getAbsolutePath());
			if (!local.exists()) {
				local = new File(System.getProperty("user.home") + "/kafka.properties");
			}
			if (local.exists()) {
				Properties tmp = new Properties();
				try (InputStream in = Files.newInputStream(local.toPath())) {
					tmp.load(in);
				} catch (Exception e) {
					logger.warn(e.getMessage());
				}
				zkBootstrap = tmp.getProperty("zookeeper.servers");
			}
		}
		return zkBootstrap;
	}

	@Override
	public void drop(String seqName) {
		try {
			curatorFramework(conf).delete().quietly().forPath(ROOT + seqName);
			logger.debug("Sequence {} dropped.", seqName);
		} catch (Exception e) {
			throw new SequencePersistenceException(e);
		}
	}

	@Override
	public Set<StoreCapability> capabilities() {
		return CAPS;
	}

	@Override
	public List<Seq> activeSequences() {
		CuratorFramework curator = curatorFramework(conf);
		try {
			List<String> children = curator.getChildren().forPath(ROOT_NODE);
			children.forEach(s -> System.err.println("- " + s));
			return children.stream().map(s -> new Seq(s.replace(ROOT, ""), curVal(s))).toList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return Collections.emptyList();
		}
	}

	public static class StoreFactory extends CqlRepositoryFactory<IItemSequencesStore>
			implements IStandaloneFactory<IItemSequencesStore> {

		@Override
		public Class<IItemSequencesStore> factoryClass() {
			return IItemSequencesStore.class;
		}

		@Override
		public IItemSequencesStore instance(BmContext context) throws ServerFault {
			return new ZkSequenceStore();
		}

	}
}
