/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.tx.outbox.service;

import java.io.File;
import java.sql.SQLException;
import java.util.Optional;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Suppliers;

import net.bluemind.config.InstallationId;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.backup.continuous.api.IBackupStoreFactory;
import net.bluemind.core.backup.continuous.api.Providers;
import net.bluemind.core.backup.continuous.dto.VersionnedItem;
import net.bluemind.core.backup.continuous.model.DefaultTopicDescriptor;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.model.RecordKey.Operation;
import net.bluemind.core.backup.continuous.model.TopicDescriptor;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueWriter;
import net.bluemind.directory.api.ReservedIds;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;
import net.bluemind.tx.outbox.api.ITxOutbox;
import net.bluemind.tx.outbox.repository.ITxOutboxRepository;

public class TxOutboxService implements ITxOutbox {

	private static final Logger logger = LoggerFactory.getLogger(TxOutboxService.class);

	private final Supplier<ITxOutboxRepository> outboxStore;
	private final Supplier<TopicDescriptor> desc;
	private final ValueWriter keyWriter;
	private final BaseContainerDescriptor cont;

	public static final boolean FORCE_DISABLE = new File("/etc/bm/tx_outbox.disabled").exists();

	private static final byte[] EMPTY = new byte[0];

	public TxOutboxService(BmContext ctx, BaseContainerDescriptor cont, DataLocation dataLocation) {
		this.cont = cont;
		this.outboxStore = Suppliers
				.memoize(() -> RepositoryProvider.instance(ITxOutboxRepository.class, ctx, dataLocation));
		this.desc = Suppliers.memoize(
				() -> DefaultTopicDescriptor.forContainer(InstallationId.getIdentifier(), cont, Optional.empty()));
		this.keyWriter = JsonUtils.writer(RecordKey.class);
	}

	public <T> long forKafka(ItemValue<T> item, ReservedIds deps, boolean deletion) {
		IBackupStoreFactory bsf = Providers.get();
		SystemState state = StateContext.getState();
		boolean paused = FORCE_DISABLE || bsf.isPaused();
		logger.trace("forKafka with bsf {}, paused: {}, state: {}", bsf, paused, state);
		if (paused || state == SystemState.CORE_STATE_UPGRADE || !bsf.leadership().isLeader()) {
			return -1L;
		}

		VersionnedItem<T> vi = new VersionnedItem<>(item, deps);
		TopicDescriptor topicDesc = desc.get();
		RecordKey key = RecordKey.forItemValue(topicDesc, item, deletion);
		TypeReference<VersionnedItem<T>> theRef = new TypeReference<VersionnedItem<T>>() {
		};
		ValueWriter valueWriter = JsonUtils.writer(theRef.getType());
		String partKey = topicDesc.partitionKey(item.uid);
		String topicSel = cleanupDomain(topicDesc.domainUid());
		byte[] k = keyWriter.write(key);
		byte[] v = valueWriter.write(vi);

		if (v != null && v.length > bsf.forContainer(cont).maximumSize()) {
			throw new ServerFault("Can't accept " + v.length
					+ " bytes in tx outbox as it is bigger than configured limit (see /etc/bm/kafka-store.conf)",
					ErrorCode.ENTITY_TOO_LARGE);
		}

		try {
			ITxOutboxRepository obStore = outboxStore.get();
			long ret = obStore.forKafka(topicSel, partKey, k, v);
			if (RecordKey.Operation.of(key) == Operation.DELETE) {
				obStore.forKafka(topicSel, partKey, keyWriter.write(key.cloneAs(Operation.UPDATE)), EMPTY);
				ret = obStore.forKafka(topicSel, partKey, keyWriter.write(key.cloneAs(Operation.CREATE)), EMPTY);
			}
			// this might go backward but we only use it for comparing with flushed values
			obStore.seqHolder().addedForBackup.set(ret);
			return ret;
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public boolean isPaused() {
		return FORCE_DISABLE || Providers.get().isPaused();
	}

	@Override
	public long lastFlushedSeq() {
		return outboxStore.get().seqHolder().flushed.get();
	}

	@Override
	public long lastOutboxSeq() {
		return outboxStore.get().seqHolder().addedForBackup.get();
	}

	private String cleanupDomain(String d) {
		return switch (d) {
		case null -> "__orphans__";
		case "global.virt" -> "__orphans__";
		default -> d;
		};
	}

}
