import { clientsClaim, type RouteHandler, RouteMatchCallback } from "workbox-core";
import { Route } from "workbox-routing";

import serviceWorker from "@bluemind/commons.light/utils/service-worker";
import { extensions } from "@bluemind/extensions";
import logger from "@bluemind/logger";
import bus from "@bluemind/service-worker-bus";
import ContainerDatasource from "@bluemind/service-worker-datasource";
import session from "@bluemind/session";

import MessageHandlerRegistry from "./ MessageHandlerRegistry";
import { ApiRouteRegistry } from "./ApiProxyPlugin/ApiRouteRegistry";
import AuthenticatedRouter from "./AuthenticatedRouter";
import BrowserData from "./BrowserData";
import ResourceListProvider from "./commons/ResourceListProvider";
import { addressbookService, mailService, subscriptionService } from "./module";
import AddressBookLocalDBProxy from "./module/addressbook/AddressBookLocalDBProxy";
//FIXME : ITINERIS
//import ConversationActionLocalDBProxy from "./module/backend.mail/ConversationActionLocalDBProxy";
import ConversationLocalDBProxy from "./module/backend.mail/ConversationLocalDBProxy";
import MailboxFolderLocalDBProxy from "./module/backend.mail/MailboxFolderLocalDBProxy";
import MailboxItemLocalDBProxy from "./module/backend.mail/MailboxItemLocalDBProxy";
import { defaultRoutes } from "./module/defaultRoutes";
import { ConnectedMessageHandler, DisconnectMessageHandler, SynchronizeMessageHandler } from "./Synchronization";
import SynchronizeQueue from "./Synchronization/SynchronizeQueue";
import SyncServiceProvider from "./Synchronization/SyncServiceProvider";
import { key } from "./Synchronization/SyncUtils";

declare let self: ServiceWorkerGlobalScope;

clientsClaim();
self.skipWaiting();

serviceWorker.addMessageListener("RESET", () => BrowserData.reset());

try {
    self.importScripts("/webapp/service-worker-extensions");
    const scripts: { path: string }[] = extensions.get("serviceworker.scripts", "script");
    self.importScripts(...scripts.map(({ path }) => "/webapp/" + path));
} catch (error) {
    logger.error("[SW][Script] Failed to import scripts");
}

//Api Client proxyfication
const handlers = extensions.get("serviceworker.handlers", "api-handler");

subscriptionService.register();
addressbookService.register();
mailService.register();
ApiRouteRegistry.register(AddressBookLocalDBProxy, 128, undefined);
ApiRouteRegistry.register(MailboxItemLocalDBProxy, 128, undefined);
ApiRouteRegistry.register(MailboxFolderLocalDBProxy, 128, undefined);
ApiRouteRegistry.register(ConversationLocalDBProxy, 128, undefined);
//FIXME : ITINERIS
//ApiRouteRegistry.register(ConversationActionLocalDBProxy, 128, undefined);
const router = new AuthenticatedRouter();

handlers.forEach((handler: any) => ApiRouteRegistry.register(handler.class, handler.priority, handler.role));
ApiRouteRegistry.routes().forEach(route => router.registerRoute(route));

defaultRoutes.setup();
const routes: {
    route: { capture: RouteMatchCallback; handler: RouteHandler; method?: Route["method"] };
    priority: number;
}[] = extensions.get("serviceworker.handlers", "route-handler");
routes.forEach(({ route }) => {
    router.registerRoute(new Route(route.capture, route.handler, route.method));
});
self.addEventListener("fetch", async (event: FetchEvent) => {
    const response = router.handleRequest({ event, request: event.request });
    if (response) {
        event.respondWith(response);
    }
});

const syncServices = extensions.get("serviceworker.handlers", "sync-handlers");
syncServices.forEach((service: any) => SyncServiceProvider.register(service.class, service.type));

const containerDBHandler = extensions.get("serviceworker.handlers", "containerdb-handlers");
containerDBHandler.forEach((container: any) => ContainerDatasource.register(container.class, container.type));

bus.channel(key).on("SYNCHRONIZE", (payload: { uid: string; type: string; priority: number }) => {
    SynchronizeQueue.queue(payload);
});

//And even Notification system to prevent multiple notifications for one event....
MessageHandlerRegistry.register(SynchronizeMessageHandler);
MessageHandlerRegistry.register(ConnectedMessageHandler);
MessageHandlerRegistry.register(DisconnectMessageHandler);

session.addEventListener("refresh", async event => {
    if (!session.isAuthenticated) {
        return;
    }
    try {
        await BrowserData.resetIfNeeded();
    } catch (e) {
        logger.error("[SW][BrowserData] Fail to reset browser data");
        logger.error(e);
    }
    serviceWorker.postMessage("REFRESH_SESSION", event.detail.sid, event.detail);
});

self.addEventListener("install", event => {
    try {
        event.waitUntil(swInstall());
    } catch (e) {
        logger.error("[Install] Failed to install service worker", e);
    }
});

async function swInstall() {
    await deleteCache();
    downloadAssets();
}

async function deleteCache() {
    try {
        await caches.delete("assets");
    } catch (e) {
        logger.log("[Install] An error occured cleaning assets from cache", e);
    }
}

async function downloadAssets() {
    try {
        await bulkFetchAndCache();
    } catch (e) {
        logger.error("[Download] Failed to download assets:", e);
        throw e;
    }
}
const bulkFetchAndCache = async () => {
    const cache = await caches.open("assets");
    const assets = await ResourceListProvider.getAssetList();
    const promises = assets.map(async url => {
        try {
            const response = await fetch(url);
            if (response.ok && !response.redirected) {
                await cache.put(url, response);
            }
        } catch (error) {
            logger.warn(`[Download] Failed tod fetch ${url}:`, error);
        }
    });

    await Promise.all(promises);
};
