/* eslint-disable @typescript-eslint/no-non-null-assertion */
import { openDB, DBSchema, IDBPDatabase, IDBPTransaction, StoreNames } from "idb";

import { Conversation, MailboxItem, MessageBody } from "@bluemind/backend.mail.api";
import { idToUid, uidToId } from "@bluemind/commons.light/backend.mail";
import { LocalItemContainerValue } from "@bluemind/commons.light/core.container";
import { SyncStatus } from "@bluemind/commons.light/model/synchronization";
import { sortedIndexBy } from "@bluemind/commons.light/utils/array";
import { isDefined } from "@bluemind/commons.light/utils/lang";
import { ItemFlag, ItemValue } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import { ContainerDB, Action, Change } from "@bluemind/service-worker-datasource";

export type MailItemLight = {
    internalId: number;
    flags: ItemFlag[];
    date: number;
    subject?: string;
    size: number;
    sender?: string;
};

export type ConversationStub = {
    conversationUid: string;
    date: number;
    size: number;
    subject?: string;
    sender?: string;
    unseen?: boolean;
    flagged?: boolean;
};

export type Counter = {
    containerUid: string;
    unseenVisible: number;
    totalVisible: number;
    total: number;
};
interface ContainerSchema extends DBSchema {
    items: {
        key: [number, string];
        value: LocalItemContainerValue<MailboxItem & { conversationId: string }>;
        indexes: {
            id: [string, string];
            containerUid: string;
            conversationId: string;
            conversationId_containerUid: [string, string];
        };
    };

    item_light: {
        key: string;
        value: MailItemLight[];
    };

    conversations_by_folder: {
        key: string;
        value: ConversationStub[];
    };
    change_set: {
        key: [number, string];
        value: Change<number>;
        indexes: { containerUid: string };
    };

    sync_status: {
        key: string;
        value: SyncStatus;
    };
}
export class MailboxRecordsDB implements ContainerDB<MailboxItem, number> {
    dbPromise: Promise<IDBPDatabase<ContainerSchema>>;
    constructor(name: string) {
        this.dbPromise = this.openDB(name);
    }
    private async openDB(name: string): Promise<IDBPDatabase<ContainerSchema>> {
        const schemaVersion = 1;
        return openDB<ContainerSchema>(`${name}`, schemaVersion, {
            upgrade(db, oldVersion) {
                logger.log(`[SW][DB] Upgrading from ${oldVersion} to ${schemaVersion}`);
                if (oldVersion < schemaVersion) {
                    logger.log("[SW][DB] Upgrading deleting existing object store");
                    for (const name of Object.values(db.objectStoreNames)) {
                        db.deleteObjectStore(name);
                    }
                }
                const store = db.createObjectStore("items", { keyPath: ["internalId", "containerUid"] });
                store.createIndex("containerUid", "containerUid");
                store.createIndex("conversationId", "value.conversationId");
                store.createIndex("conversationId_containerUid", ["value.conversationId", "containerUid"]);
                db.createObjectStore("item_light");
                db.createObjectStore("change_set", { keyPath: ["uid", "containerUid"] }).createIndex(
                    "containerUid",
                    "containerUid"
                );
                db.createObjectStore("sync_status");
                db.createObjectStore("conversations_by_folder");
            },
            blocking: async () => {
                await this.close();
                this.dbPromise = this.openDB(name);
            }
        });
    }

    async close(): Promise<void> {
        (await this.dbPromise).close();
    }

    async putItems(items: ItemValue<MailboxItem & { conversationId: string }>[], containerUid: string) {
        const tx = (await this.dbPromise).transaction(
            ["items", "item_light", "change_set", "sync_status", "conversations_by_folder"],
            "readwrite"
        );
        const conversationUid = new Set<string>();
        for (const item of items) {
            tx.objectStore("items").put({ ...item, containerUid });
            tx.objectStore("change_set").put({ uid: item.internalId as number, containerUid, action: Action.UPDATE });
            conversationUid.add(idToUid(item.value.conversationId));
        }
        await this.putItemLight(items, containerUid, tx);

        const syncStatus = await this.getOrDefaultSyncStatus(containerUid, tx);
        syncStatus.stale = true;
        tx.objectStore("sync_status").put(syncStatus, containerUid);
        await this.refreshConversationStubs(conversationUid, containerUid, tx);
        await tx.done;
    }

    async putItemsAndCommit(items: ItemValue<MailboxItem & { conversationId: string }>[], containerUid: string) {
        const tx = (await this.dbPromise).transaction(["items", "item_light", "conversations_by_folder"], "readwrite");
        const conversationUids = new Set<string>();
        for (const item of items) {
            tx.objectStore("items").put({ ...item, containerUid });
            conversationUids.add(idToUid(item.value.conversationId));
        }
        await this.putItemLight(items, containerUid, tx);
        await this.refreshConversationStubs(conversationUids, containerUid, tx);
        await tx.done;
    }

    async deleteItems(
        ids: number[],
        containerUid: string,
        _tx?: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const tx =
            _tx ||
            (await this.dbPromise).transaction(["items", "item_light", "change_set", "sync_status"], "readwrite");
        for (const id of ids) {
            tx.objectStore("items").delete([id, containerUid]);
            tx.objectStore("change_set").put({ uid: id, containerUid, action: Action.DELETE });
        }
        await this.deleteItemLight(ids, containerUid, tx);
        const syncStatus = await this.getOrDefaultSyncStatus(containerUid, tx);
        syncStatus.stale = true;
        tx.objectStore("sync_status").put(syncStatus, containerUid);

        if (!_tx) {
            await tx.done;
        }
    }

    async deleteItemsAndCommit(
        ids: number[],
        containerUid: string,
        _tx?: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const tx =
            _tx || (await this.dbPromise).transaction(["items", "item_light", "conversations_by_folder"], "readwrite");
        const store = tx.objectStore("items");
        const results = await Promise.all(ids.map(id => store.get([id, containerUid])));
        for (const id of ids) {
            tx.objectStore("items").delete([id, containerUid]);
        }
        await this.deleteItemLight(ids, containerUid, tx);

        const conversationUids: Set<string> = new Set(
            results.flatMap(item => (item?.value?.conversationId ? [idToUid(item.value.conversationId)] : []))
        );
        await this.refreshConversationStubs(conversationUids, containerUid, tx);

        if (!_tx) {
            await tx.done;
        }
    }

    async getItems(
        ids: number[],
        containerUid: string
    ): Promise<ItemValue<MailboxItem & { conversationId: string }>[]> {
        const tx = (await this.dbPromise).transaction("items", "readonly");
        const store = tx.objectStore("items");
        const results = await Promise.all(ids.map(id => store.get([id, containerUid])));
        await tx.done;
        return results.filter(isDefined);
    }

    async getAllItems(containerUid: string): Promise<ItemValue<MailboxItem & { conversationId: string }>[]> {
        return (await this.dbPromise).getAllFromIndex("items", "containerUid", containerUid);
    }
    async getChangeSet(containerUid: string) {
        return (await this.dbPromise).getAllFromIndex("change_set", "containerUid", containerUid);
    }

    async commit(ids: number[], containerUid: string) {
        const tx = (await this.dbPromise).transaction("change_set", "readwrite");
        for (const id of ids) {
            tx.objectStore("change_set").delete([id, containerUid]);
        }
        await tx.done;
    }

    async getSyncStatus(containerUid: string) {
        return (await this.dbPromise).get("sync_status", containerUid);
    }
    async setSyncStatus(syncStatus: SyncStatus, containerUid: string) {
        (await this.dbPromise).put("sync_status", syncStatus, containerUid);
    }
    async getAllItemLight(containerUid: string): Promise<MailItemLight[]> {
        return (await (await this.dbPromise).get("item_light", containerUid)) || [];
    }
    async getAllItemLightByFolders(containerUids: string[]): Promise<Map<string, MailItemLight[]>> {
        const db = await this.dbPromise;
        const tx = db.transaction(["item_light"], "readonly");
        const store = tx.objectStore("item_light");

        const getItemLightsPromises = containerUids.map(uid =>
            store.get(uid).then(result => ({ uid, items: result || [] }))
        );

        const results = await Promise.all(getItemLightsPromises);

        const itemLightsByFolder = results.reduce((map, { uid, items }) => {
            map.set(uid, items);
            return map;
        }, new Map<string, MailItemLight[]>());

        await tx.done;
        return itemLightsByFolder;
    }
    private async putItemLight(
        items: ItemValue<MailboxItem>[],
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const lights: Array<MailItemLight> = (await tx.objectStore("item_light").get(containerUid)) || [];
        try {
            items.forEach(mail => {
                const light = toLight(mail);
                const index = sortedIndexBy(lights, light, "internalId");
                const isPresent = lights[index]?.internalId === light.internalId;
                lights.splice(index, isPresent ? 1 : 0, light);
            });
        } catch (error) {
            tx.abort();
            logger.error("[MailboxRecordsDB] fail to convert items to light", items, containerUid);
        }
        tx.objectStore("item_light").put(lights, containerUid);
    }

    private async deleteItemLight(
        items: number[],
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const lights: Array<MailItemLight> = (await tx.objectStore("item_light").get(containerUid)) || [];
        items.forEach(id => {
            const dummy = { internalId: id, flags: [], date: 0, subject: "", size: 0, sender: "" };
            const index = sortedIndexBy(lights, dummy, "internalId");
            if (lights[index]?.internalId === id) {
                lights.splice(index, 1);
            }
        });
        tx.objectStore("item_light").put(lights, containerUid);
    }

    private async getOrDefaultSyncStatus(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], IDBTransactionMode>
    ) {
        return (
            (await tx.objectStore("sync_status").get(containerUid)) || {
                stale: false,
                version: 0
            }
        );
    }
    async getConversations(uids: string[], folders: Set<string>): Promise<Conversation[]> {
        const db = await this.dbPromise;
        const tx = db.transaction(["items"], "readonly");

        const results = await Promise.all(
            uids.map(async uid => {
                return toConversation(await this.getMailsByConversation(uid, tx, folders));
            })
        );
        await tx.done;
        return results;
    }

    async getConversationMessageRefs(conversationUids: string[], folderUid: string): Promise<Conversation[]> {
        const db = await this.dbPromise;
        const tx = db.transaction(["items"], "readonly");

        const results = await Promise.all(
            conversationUids.map(async uid => {
                return toConversation(await this.getMailsByConversation(uid, tx, new Set([folderUid])));
            })
        );

        await tx.done;
        return results;
    }

    private async getMailsByConversation(
        uid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readonly" | "readwrite">,
        folders: Set<string>
    ): Promise<LocalItemContainerValue<MailboxItem & { conversationId: string }>[]> {
        const store = tx.objectStore("items");
        if (folders.size === 1) {
            const index = store.index("conversationId_containerUid");
            return index.getAll([uidToId(uid), folders.values().next().value!]);
        } else {
            const index = store.index("conversationId");
            return (await index.getAll(uidToId(uid))).filter(({ containerUid }) => folders.has(containerUid));
        }
    }

    async getConversationByFolder(containerUid: string): Promise<ConversationStub[]> {
        const tx = (await this.dbPromise).transaction(["conversations_by_folder"], "readonly");
        const convs = (await tx.objectStore("conversations_by_folder").get(containerUid)) || [];

        await tx.done;
        return convs;
    }

    async deleteConversations(conversationUids: string[], containerUid: string): Promise<void> {
        const conversationUidsSet = new Set(conversationUids);
        const tx = (await this.dbPromise).transaction(
            ["items", "conversations_by_folder", "item_light", "change_set", "sync_status"],
            "readwrite"
        );

        const allMessages = await Promise.all(
            conversationUids.map(uid => this.getMailsByConversation(uid, tx, new Set([containerUid])))
        );
        const allInternalIds = allMessages.flat().map(message => message.internalId);
        await this.deleteItems(allInternalIds as number[], containerUid, tx);

        await this.refreshConversationStubs(conversationUidsSet, containerUid, tx);
        await tx.done;
        return;
    }

    private async refreshConversationStubs(
        conversationUids: Set<string>,
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const stubs = ((await tx.objectStore("conversations_by_folder").get(containerUid)) || []) as ConversationStub[];
        for (const uid of conversationUids) {
            const stub = toConversationsStubs(await this.getMailsByConversation(uid, tx, new Set([containerUid])), uid);
            const index = sortedIndexBy(stubs, stub, "conversationUid");
            const isPresent = stubs[index]?.conversationUid === stub.conversationUid;
            stubs.splice(index, isPresent ? 1 : 0, ...(stub.size > 0 ? [stub] : []));
        }
        tx.objectStore("conversations_by_folder").put(stubs, containerUid);
    }

    async reset(containerUid: string): Promise<void> {
        const tx = (await this.dbPromise).transaction(
            ["items", "item_light", "change_set", "sync_status", "conversations_by_folder"],
            "readwrite"
        );
        await Promise.all([
            this.resetItems(containerUid, tx),
            this.resetChangeSet(containerUid, tx),
            this.resetSyncStatus(containerUid, tx)
        ]);
        await tx.done;
    }

    private async resetItems(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const items = await tx.objectStore("items").index("containerUid").getAll(containerUid);

        const internalIds = items.flatMap(item => item.internalId) || [];
        for (const id of internalIds as number[]) {
            tx.objectStore("items").delete([id, containerUid]);
        }
        tx.objectStore("conversations_by_folder").delete(containerUid);
        tx.objectStore("item_light").delete(containerUid);
    }

    private async resetChangeSet(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const changeSetStore = tx.objectStore("change_set");
        const changeSetKeys = await changeSetStore.index("containerUid").getAllKeys(containerUid);
        await Promise.all(changeSetKeys.map(key => changeSetStore.delete(key)));
    }

    private async resetSyncStatus(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        tx.objectStore("sync_status").delete(containerUid);
    }
}

function toConversationsStubs(mails: ItemValue<MailboxItem>[], conversationUid: string): ConversationStub {
    return mails.reduce(
        (stub, mail) =>
            ({
                conversationUid,
                //FIXME : Sanitize subject ?
                subject: stub.subject || getSubject(mail),
                unseen: stub.unseen || !mail.flags?.includes(ItemFlag.Seen),
                flagged: stub.flagged || mail.flags?.includes(ItemFlag.Important),
                size: stub.size > mail.value.body.size! ? stub.size : mail.value.body.size,
                //FIXME : get first originator ?
                sender: stub.sender || getOriginator(mail),
                date: stub.date > mail.value.body.date! ? stub.date : mail.value.body.date
            } as ConversationStub),
        {
            conversationUid,
            size: 0,
            date: 0
        } as ConversationStub
    );
}

function toLight(mail: ItemValue<MailboxItem>): MailItemLight {
    return {
        internalId: mail.internalId as number,
        flags: mail.flags as ItemFlag[],
        date: mail.value.body.date as number,
        subject: mail.value.body.subject?.toLowerCase().replace(/^(\W*|re\s*:)*/i, ""),
        size: mail.value.body.size as number,
        sender: mail.value.body.recipients
            ?.find((recipient: any) => recipient.kind === "Originator")
            ?.address?.toLowerCase()
    };
}

function toConversation(mails: LocalItemContainerValue<MailboxItem & { conversationId: string }>[]): Conversation {
    return mails.reduce(
        (conversation, mail) => ({
            messageRefs: [...(conversation.messageRefs || []), toRef(mail)],
            conversationUid: idToUid(mail.value.conversationId)
        }),
        {} as Conversation
    );
}

function toRef(mail: LocalItemContainerValue<MailboxItem>) {
    return {
        folderUid: mail.containerUid,
        itemId: mail.internalId,
        date: mail.value.body.date
    };
}

function getSubject(mail: ItemValue<MailboxItem>) {
    return mail.value.body.subject;
}
function getOriginator(mail: ItemValue<MailboxItem>) {
    return mail.value.body.recipients?.find((recipient: any) => recipient.kind === MessageBody.RecipientKind.Originator)
        ?.address;
}
