import FDBFactory from "fake-indexeddb/lib/FDBFactory";

import { MailboxItem } from "@bluemind/backend.mail.api";

import conversation from "@bluemind/commons.light/backend.mail/conversation";
import { ItemFlag, ItemValue } from "@bluemind/core.container.api";
import { Action } from "@bluemind/service-worker-datasource";

import { MailboxRecordsDB } from "./MailboxRecordsDB";

vi.mock("@bluemind/logger", () => ({
    default: {
        error: vi.fn(),
        log: vi.fn()
    }
}));

const containerUid = "containerUid";
describe("ContainerDB Interface Tests", () => {
    let mockContainerDB: MailboxRecordsDB;
    const testItems: (ItemValue<MailboxItem & { conversationId: string }> & { containerUid: string })[] = [
        {
            containerUid,
            uid: "1",
            internalId: 1,
            flags: [ItemFlag.Important],
            value: {
                conversationId: "1",
                body: { date: 1, recipients: [{ kind: "Originator", address: "A" }], subject: "Subject", size: 1 }
            }
        },
        {
            containerUid,
            uid: "2",
            internalId: 2,
            flags: [ItemFlag.Important],
            value: {
                conversationId: "2",
                body: { date: 2, recipients: [{ kind: "Originator", address: "B" }], subject: "Subject B", size: 2 }
            }
        }
    ];

    const testLightItems = [
        {
            date: 1,
            flags: ["Important"],
            internalId: 1,
            sender: "a",
            size: 1,
            subject: "subject"
        },
        {
            date: 2,
            flags: ["Important"],
            internalId: 2,
            sender: "b",
            size: 2,
            subject: "subject b"
        },
        {
            date: 3,
            flags: ["Seen"],
            internalId: 3,
            sender: "c",
            size: 3,
            subject: "subject c"
        }
    ];

    const testIds = [1, 2];

    beforeEach(async () => {
        global.indexedDB = new FDBFactory();
        mockContainerDB = new MailboxRecordsDB("containerName");
        await mockContainerDB.setSyncStatus({ version: 0, stale: false }, containerUid);
        vi.clearAllMocks();
    });

    it("should Put items and set changeset", async () => {
        await mockContainerDB.putItems(testItems, containerUid);

        const items = await mockContainerDB.getAllItems(containerUid);
        const changeSet = await mockContainerDB.getChangeSet(containerUid);

        expect(changeSet[0].action).toBe(Action.UPDATE);
        expect(items.length).toBe(testItems.length);
    });
    it("Put items without updating changeset", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        const items = await mockContainerDB.getAllItems(containerUid);
        const changeSet = await mockContainerDB.getChangeSet(containerUid);

        expect(changeSet.length).toBe(0);
        expect(items.length).toBe(testItems.length);
    });

    it("Delete items and set changeset", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        await mockContainerDB.deleteItems(testIds, containerUid);

        const items = await mockContainerDB.getAllItems(containerUid);
        const changeSet = await mockContainerDB.getChangeSet(containerUid);
        expect(changeSet[0].action).toBe(Action.DELETE);
        expect(items.length).toBe(0);
    });

    it("Delete items without updating changeset", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        await mockContainerDB.deleteItemsAndCommit(testIds, containerUid);

        const items = await mockContainerDB.getAllItems(containerUid);
        const changeSet = await mockContainerDB.getChangeSet(containerUid);
        expect(changeSet.length).toBe(0);
        expect(items.length).toBe(0);
    });

    it("Correct items are fetch", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);
        await mockContainerDB.putItemsAndCommit(testItems, "other" + containerUid);

        const items = await mockContainerDB.getItems(testIds, containerUid);

        expect(items).toStrictEqual(testItems);
    });

    it("Get all items", async () => {
        testItems.push({
            containerUid,
            uid: "3",
            internalId: 3,
            flags: [ItemFlag.Seen],
            value: {
                conversationId: "3",
                body: { date: 3, recipients: [{ kind: "Originator", address: "C" }], subject: "Subject C", size: 3 }
            }
        });

        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        const items = await mockContainerDB.getAllItems(containerUid);

        expect(items).toStrictEqual(testItems);
    });

    it("Get all light items", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        const items = await mockContainerDB.getAllItemLight(containerUid);

        expect(items).toStrictEqual(testLightItems);
    });

    it("Get conversations", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        const items = await mockContainerDB.getConversations(["1", "2"]);

        expect(items).toStrictEqual([
            {
                conversationUid: "1",
                messageRefs: [
                    {
                        date: 1,
                        folderUid: containerUid,
                        itemId: 1
                    }
                ]
            },
            {
                conversationUid: "2",
                messageRefs: [
                    {
                        date: 2,
                        folderUid: containerUid,
                        itemId: 2
                    }
                ]
            }
        ]);
    });

    it("Get conversations message refs", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        const items = await mockContainerDB.getConversationMessageRefs(["1", "2"], containerUid);

        expect(items).toStrictEqual([
            {
                conversationUid: "1",
                messageRefs: [
                    {
                        date: 1,
                        folderUid: containerUid,
                        itemId: 1
                    }
                ]
            },
            {
                conversationUid: "2",
                messageRefs: [
                    {
                        date: 2,
                        folderUid: containerUid,
                        itemId: 2
                    }
                ]
            }
        ]);
    });

    it("Get conversations by folder", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        const items = await mockContainerDB.getConversationByFolder(containerUid);

        expect(items).toStrictEqual([
            {
                conversationUid: "1",
                date: 1,
                flagged: true,
                sender: "A",
                size: 1,
                subject: "Subject",
                unseen: true
            },
            {
                conversationUid: "2",
                date: 2,
                flagged: true,
                sender: "B",
                size: 2,
                subject: "Subject B",
                unseen: true
            },
            {
                conversationUid: "3",
                date: 3,
                flagged: false,
                sender: "C",
                size: 3,
                subject: "Subject C",
                unseen: false
            }
        ]);
    });

    it("Delete conversations by folder", async () => {
        await mockContainerDB.putItemsAndCommit(testItems, containerUid);

        await mockContainerDB.deleteConversations(["1", "2"], containerUid);

        const items = await mockContainerDB.getAllItems(containerUid);
        expect(items).toStrictEqual([testItems[2]]);
    });

    describe("reset", () => {
        it("Reset items", async () => {
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);

            await mockContainerDB.reset(containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);
            expect(items).toStrictEqual([]);
            const sync = await mockContainerDB.getSyncStatus(containerUid);
            expect(sync).toStrictEqual(undefined);
            const changeSet = await mockContainerDB.getChangeSet(containerUid);
            expect(changeSet).toStrictEqual([]);
        });
    });
});
