import { describe, it, expect, vi, beforeEach } from "vitest";

import { AddressBookClient } from "@bluemind/addressbook.api";
import ContainerDatasource, { Action } from "@bluemind/service-worker-datasource";

import { setupMockDatasource } from "../../testUtils/MockDatasource";

import { setupService } from "../../testUtils/MockService";

import AddressbookSyncService from "./AddressbookSyncService";

vi.mock("@bluemind/addressbook.api", () => ({
    AddressBookClient: vi.fn().mockImplementation(() => ({
        changeset: vi.fn(),
        multipleGet: vi.fn(),
        updates: vi.fn()
    }))
}));

vi.mock("@bluemind/service-worker-datasource", () => ({
    default: {
        retrieve: vi.fn()
    },
    Action: {
        CREATE: 0,
        UPDATE: 1,
        DELETE: 2
    }
}));

const createMockChangeset = (version = 1) => ({
    created: ["id3", "id4"],
    updated: ["id1", "id2"],
    deleted: ["id5", "id6"],
    version
});

describe("AddressbookSyncService", () => {
    let service: AddressbookSyncService;
    let mockClient: any;
    let mockDb: any;

    beforeEach(() => {
        vi.clearAllMocks();
        ({ mockDb } = setupMockDatasource({
            getItems: vi.fn().mockResolvedValue([
                { uid: "id1", value: "id1", version: 1 },
                { uid: "id2", value: "id2", version: 1 }
            ]),
            getChangeSet: vi.fn().mockResolvedValue([
                { action: Action.CREATE, uid: "id1" },
                { action: Action.CREATE, uid: "id2" },
                { action: Action.UPDATE, uid: "id3" },
                { action: Action.DELETE, uid: "id4" },
                { action: Action.DELETE, uid: "id5" }
            ])
        }));
        (ContainerDatasource.retrieve as any).mockReturnValue(mockDb);
        ({ service, mockClient } = setupService(AddressbookSyncService, "test-uid", "test-sid"));
    });

    describe("constructor", () => {
        it("should initialize with correct parameters", () => {
            expect(AddressBookClient).toHaveBeenCalledWith("test-sid", "test-uid");
            expect(service.client).toBeDefined();
            expect(service.db).toBeDefined();
        });
    });

    describe("getRemoteChangeSet method", () => {
        it("should fetch remote changes and merge created/updated arrays", async () => {
            const mockChangeset = createMockChangeset();
            mockClient.changeset.mockResolvedValue(mockChangeset);

            const result = await service.getRemoteChangeSet(2);

            expect(mockClient.changeset).toHaveBeenCalledWith(2);
            expect(result).toEqual({
                updated: ["id4", "id3", "id2", "id1"],
                deleted: ["id5", "id6"],
                version: 1
            });
        });
    });

    describe("getRemoteItems method", () => {
        it("should fetch remote items in chunks of 200", async () => {
            const generatedIds = Array.from({ length: 250 }, (_, i) => `id${i}`);
            const mockResponse1 = {
                uid: "id0",
                value: "id0",
                version: 1
            };
            const mockResponse2 = {
                uid: "id1",
                value: "id1",
                version: 1
            };

            mockClient.multipleGet.mockResolvedValueOnce([mockResponse1]);
            mockClient.multipleGet.mockResolvedValueOnce([mockResponse2]);

            const result = await service.getRemoteItems(generatedIds);

            expect(mockClient.multipleGet).toHaveBeenCalledTimes(2);
            expect(mockClient.multipleGet.mock.calls[0][0]).toHaveLength(200);
            expect(mockClient.multipleGet.mock.calls[1][0]).toHaveLength(50);
            expect(result).toEqual([mockResponse1, mockResponse2]);
        });

        it("should return empty array for empty id list", async () => {
            const result = await service.getRemoteItems([]);

            expect(mockClient.multipleGet).not.toHaveBeenCalled();
            expect(result).toEqual([]);
        });

        it("should ignore null items", async () => {
            const mockResponse1 = {
                uid: "id0",
                value: "id0",
                version: 1
            };

            mockClient.multipleGet.mockResolvedValueOnce([mockResponse1, null]);

            const result = await service.getRemoteItems(["id0", "id1"]);

            expect(result).toEqual([mockResponse1]);
        });
    });

    describe("updateRemote method", () => {
        it("should update remote with local changes and commit response", async () => {
            const changeSet = {
                updated: ["id1", "id2"],
                deleted: ["id3", "id4"],
                version: 0
            };

            const mockUpdateResponse = {
                added: [{ uid: "id5" }],
                updated: [{ uid: "id6" }],
                removed: [{ uid: "id7" }]
            };

            mockClient.updates.mockResolvedValue(mockUpdateResponse);

            await service.updateRemote(changeSet, "test-uid");

            expect(mockDb.getItems).toHaveBeenCalledWith(changeSet.updated, "test-uid");
            expect(mockClient.updates).toHaveBeenCalledWith({
                added: [
                    { uid: "id1", value: "id1" },
                    { uid: "id2", value: "id2" }
                ],
                modify: [
                    { uid: "id1", value: "id1" },
                    { uid: "id2", value: "id2" }
                ],
                delete: [{ uid: "id3" }, { uid: "id4" }]
            });
            expect(mockDb.commit).toHaveBeenCalledWith([{ uid: "id5" }, { uid: "id6" }, { uid: "id7" }], "test-uid");
        });
    });

    describe("getLocalChangeSet method", () => {
        it("should categorize local changes by action type", async () => {
            const result = await service.getLocalChangeSet("test-uid");

            expect(mockDb.getChangeSet).toHaveBeenCalled();
            expect(result).toEqual({
                created: ["id1", "id2"],
                updated: ["id3"],
                deleted: ["id4", "id5"],
                version: 0
            });
        });
    });

    it("Reset DB", async () => {
        await service.resetData("test");

        expect(mockDb.reset).toHaveBeenCalledWith("test");
    });
});
