import { VCard } from "@bluemind/addressbook.api";

import { relaxedSynchronize } from "../../Synchronization/SyncUtils";
import { setupDatasourceError, setupMockDatasource } from "../../testUtils/MockDatasource";

import AddressBookLocalDBProxy from "./AddressBookLocalDBProxy";

vi.mock("@bluemind/logger", () => ({
    default: {
        error: vi.fn()
    }
}));
vi.mock("../../Synchronization/SyncUtils", () => ({
    relaxedSynchronize: vi.fn(),
    isLocal: vi.fn().mockResolvedValue(true),
    isSynchronized: vi.fn().mockResolvedValue(true)
}));
const mockNext = vi.fn().mockResolvedValue(undefined);

const setupProxy = () => new AddressBookLocalDBProxy("api-key", "addressbook-uid");

const setupMockSynchronize = () => {
    const mockRelaxedSynchronize = vi.mocked(relaxedSynchronize);
    mockRelaxedSynchronize.mockResolvedValue(undefined);
    return mockRelaxedSynchronize;
};

describe("AddressBookLocalDBProxy", () => {
    let proxy: any;
    let mockRelaxedSynchronize: any;
    beforeEach(() => {
        vi.clearAllMocks();
        proxy = setupProxy();
        proxy.next = mockNext;
        mockRelaxedSynchronize = setupMockSynchronize();
    });

    describe("update method", () => {
        const fakeItem: VCard = { kind: "group" };

        test("should use proxy for successful update", async () => {
            setupMockDatasource({
                putItems: vi.fn().mockResolvedValue("")
            });

            await proxy.update("1", fakeItem);

            expect(mockRelaxedSynchronize).toHaveBeenCalledOnce();
        });

        test("should fallback to next handler when datasource fails", async () => {
            setupDatasourceError();

            await proxy.update("1", fakeItem);

            expect(proxy.next).toHaveBeenCalledOnce();
        });
    });

    describe("delete method", () => {
        test("should use proxy for successful delete", async () => {
            const mockRelaxedSynchronize = setupMockSynchronize();
            setupMockDatasource({
                deleteItems: vi.fn().mockResolvedValue("")
            });

            await proxy.delete("1");

            expect(mockRelaxedSynchronize).toHaveBeenCalledOnce();
        });

        test("should fallback to next handler when datasource fails", async () => {
            setupDatasourceError();
            proxy.next = mockNext;

            await proxy.delete("1");

            expect(proxy.next).toHaveBeenCalledOnce();
        });
    });

    describe("getComplete method", () => {
        test("should retrieve items through proxy", async () => {
            setupMockSynchronize();
            const { mockDb } = setupMockDatasource({
                getItems: vi.fn().mockResolvedValue([1, 2, 3])
            });

            await proxy.getComplete("1");

            expect(mockDb.getItems).toHaveBeenCalledOnce();
        });

        test("should fallback when no content found", async () => {
            setupMockSynchronize();
            setupMockDatasource({
                getItems: vi.fn().mockResolvedValue([])
            });

            await proxy.getComplete("1");

            expect(proxy.next).toHaveBeenCalledOnce();
        });

        test("should fallback to next handler when datasource fails", async () => {
            setupDatasourceError();

            await proxy.getComplete("1");

            expect(proxy.next).toHaveBeenCalledOnce();
        });
    });

    describe("getCompleteById method", () => {
        test("should retrieve items by ID through proxy", async () => {
            setupMockSynchronize();
            const { mockDb } = setupMockDatasource({
                getAllItems: vi.fn().mockResolvedValue([{ internalId: 1 }])
            });

            await proxy.getCompleteById(1);

            expect(mockDb.getAllItems).toHaveBeenCalledOnce();
        });

        test("should fallback when no content found", async () => {
            setupMockSynchronize();
            setupMockDatasource({
                getAllItems: vi.fn().mockResolvedValue([])
            });

            await proxy.getCompleteById(1);

            expect(proxy.next).toHaveBeenCalledOnce();
        });

        test("should fallback to next handler when datasource fails", async () => {
            setupDatasourceError();

            await proxy.getCompleteById(1);

            expect(proxy.next).toHaveBeenCalledOnce();
        });
    });

    describe("multipleGet method", () => {
        test("should retrieve multiple items through proxy", async () => {
            setupMockSynchronize();
            const { mockDb } = setupMockDatasource({
                getItems: vi.fn().mockResolvedValue([1])
            });

            await proxy.multipleGet(["1"]);

            expect(mockDb.getItems).toHaveBeenCalledOnce();
        });

        test("should fallback to next handler when datasource fails", async () => {
            setupDatasourceError();

            await proxy.multipleGet(["1"]);

            expect(proxy.next).toHaveBeenCalledOnce();
        });
    });

    describe("multipleGetById method", () => {
        test("should retrieve multiple items by ID through proxy", async () => {
            setupMockSynchronize();
            const { mockDb } = setupMockDatasource({
                getAllItems: vi.fn().mockResolvedValue([{ internalId: 1 }])
            });
            await proxy.multipleGetById([1]);

            expect(mockDb.getAllItems).toHaveBeenCalledOnce();
        });

        test("should fallback to next handler when datasource fails", async () => {
            setupDatasourceError();

            await proxy.multipleGetById([1]);

            expect(proxy.next).toHaveBeenCalledOnce();
        });
    });
});
