package net.bluemind.memory.pool.mmap;

import java.io.InputStream;
import java.lang.foreign.MemorySegment;
import java.util.concurrent.atomic.AtomicBoolean;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.Unpooled;
import net.bluemind.memory.pool.api.IChunk;

public class Chunk implements IChunk {
	protected final Segment parent;
	protected final long startOffset;
	protected final int size;
	protected final AtomicBoolean released;

	public Chunk(Segment parent, long startOffset, int size) {
		this.parent = parent;
		this.startOffset = startOffset;
		this.size = size;
		this.released = new AtomicBoolean(false);
	}

	public void release() {
		if (released.compareAndSet(false, true)) {
			parent.releaseChunk(startOffset, size);
		}
	}

	public boolean isReleased() {
		return released.get();
	}

	public ByteBuf openBuffer() {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}

		MemorySegment segment = parent.getMemorySegment();
		MemorySegment slice = segment.asSlice(startOffset, getSize());

		java.nio.ByteBuffer nioBuffer = slice.asByteBuffer();

		return Unpooled.wrappedBuffer(nioBuffer).asReadOnly();
	}

	public InputStream openStream() {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}
		ByteBuf in = openBuffer();
		return new ByteBufInputStream(in, true);
	}

	public int getSize() {
		return size;
	}

	public long getStartOffset() {
		return startOffset;
	}

	public Segment getParent() {
		return parent;
	}

	@Override
	public String toString() {
		return "Chunk{seg: " + parent.getId() + ", offset: " + startOffset + ", size: " + getSize() + ", released: "
				+ released.get() + "}";
	}
}