/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.user;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import com.github.freva.asciitable.AsciiTable;
import com.github.freva.asciitable.Column;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;
import net.bluemind.user.api.UserProperties;
import picocli.CommandLine.ArgGroup;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "mfa", description = "Manage user MFA configuration")
public class UserMFACommand implements ICmdLet, Runnable {

	private record UserDomainUid(String domainUid, ItemValue<User> user) {
	}

	private record UserOtpStatus(String domainUid, String userUid, String userEmail, boolean otpRequired,
			boolean otpConfigured, boolean passwordDisallowed) {
	}

	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("user");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return UserMFACommand.class;
		}
	}

	private CliContext ctx;
	private CliUtils cliUtils;

	@ArgGroup(exclusive = true, multiplicity = "1", heading = "User options%n")
	public UserOpts userOpts;

	public static class UserOpts {
		@Option(names = "--email", arity = "1..*", description = "User email%n")
		public String[] emails;

		@ArgGroup(exclusive = false, heading = "User and domain UIDs%n")
		public UserUids uids;
	}

	public static class UserUids {
		@Option(names = "--user-uid", arity = "1..*", required = true, description = "User UID%n")
		public String[] uids;

		@Option(names = "--domain-uid", required = true, description = "User domain UID%n")
		public String domainUid;
	}

	@ArgGroup(exclusive = true, multiplicity = "1", heading = "MFA options%n")
	public MfaOpts mfaOpts;

	public static class MfaOpts {
		@Option(names = "--otp-required", negatable = true, description = "Ensures that the user has set up OTP, or force set up on next login%n")
		public Boolean otpRequired;

		@Option(names = "--allow-password", negatable = true, description = "(Dis)Allow password authentication for all protocols other than web access (IMAP/SMTP/MAPI/EAS...)%n")
		public Boolean allowPassword;

		@ArgGroup(exclusive = false, heading = "Display user MFA status%n")
		public StatusOpts statusOpts;
	}

	public static class StatusOpts {
		@Option(names = "--status", required = true, description = "Display user MFA status%n")
		public boolean status = false;

		@Option(names = "--json", required = false, defaultValue = "false", description = "Display user MFA status using Json format\nTable format otherwise")
		public boolean json;
	}

	@Override
	public void run() {
		List<UserDomainUid> userDomainUids = getUser();

		if (mfaOpts.otpRequired != null) {
			otpUserManageRequired(mfaOpts.otpRequired, userDomainUids);
			return;
		}

		if (mfaOpts.allowPassword != null) {
			otpUserManagePasswordAuthentication(mfaOpts.allowPassword, userDomainUids);
			return;
		}

		if (mfaOpts.statusOpts != null) {
			otpUserStatus(userDomainUids);
		}
	}

	private void otpUserStatus(List<UserDomainUid> userDomainUids) {
		List<UserOtpStatus> usersOtpValue = userDomainUids.stream().map(userDomainUid -> new UserOtpStatus(
				userDomainUid.domainUid, userDomainUid.user.uid,
				userDomainUid.domainUid().equals("global.virt") ? "admin0@global.virt"
						: userDomainUid.user.value.defaultEmail().address,
				Boolean.valueOf(userDomainUid.user.value.properties.get(UserProperties.OTP_REQUIRED.name())),
				ctx.adminApi().instance(IInternalCredentials.class, userDomainUid.domainUid)
						.getUserCredentials(userDomainUid.user.uid).stream().anyMatch(c -> c.type.equals("otp")),
				Boolean.valueOf(userDomainUid.user.value.properties.get(UserProperties.NO_PASSWD_AUTH.name()))))
				.toList();

		ctx.info(mfaOpts.statusOpts.json ? JsonUtils.asString(usersOtpValue)
				: AsciiTable.getTable(usersOtpValue,
						Arrays.asList(new Column().header("Email").with(userOtpValue -> userOtpValue.userEmail),
								new Column().header("OTP required")
										.with(userOtpValue -> Boolean.toString(userOtpValue.otpRequired)),
								new Column().header("OTP configured")
										.with(userOtpValue -> Boolean.toString(userOtpValue.otpConfigured)),
								new Column().header("Password allowed")
										.with(userOtpValue -> Boolean.toString(!userOtpValue.passwordDisallowed)))));
	}

	private void otpUserManageRequired(boolean otpRequired, List<UserDomainUid> userDomainUids) {
		userDomainUids.forEach(userDomainUid -> {
			if (otpRequired) {
				userDomainUid.user.value.properties.put(UserProperties.OTP_REQUIRED.name(), Boolean.TRUE.toString());
			} else {
				userDomainUid.user.value.properties.remove(UserProperties.OTP_REQUIRED.name());
			}

			ctx.adminApi().instance(IUser.class, userDomainUid.domainUid).update(userDomainUid.user.uid,
					userDomainUid.user.value);
		});
	}

	private void otpUserManagePasswordAuthentication(boolean allowPassword, List<UserDomainUid> userDomainUids) {
		userDomainUids.forEach(userDomainUid -> {
			if (allowPassword) {
				userDomainUid.user.value.properties.remove(UserProperties.NO_PASSWD_AUTH.name());
			} else {
				userDomainUid.user.value.properties.put(UserProperties.NO_PASSWD_AUTH.name(), Boolean.TRUE.toString());
			}

			ctx.adminApi().instance(IUser.class, userDomainUid.domainUid).update(userDomainUid.user.uid,
					userDomainUid.user.value);
		});
	}

	private List<UserDomainUid> getUser() {
		List<UserDomainUid> userDomainUids = new ArrayList<>();

		if (userOpts.uids != null) {
			for (String userUid : userOpts.uids.uids) {
				userDomainUids.add(getUser(userOpts.uids.domainUid, userUid));
			}
			return userDomainUids;
		}

		for (String email : userOpts.emails) {
			userDomainUids.add(getUser(cliUtils.getDomainUidByEmail(email),
					email.equals("admin0@global.virt") ? "admin0_global.virt" : cliUtils.getUserUidByEmail(email)));
		}

		return userDomainUids;
	}

	private UserDomainUid getUser(String domainUid, String userUid) {
		ItemValue<User> user = null;
		try {
			user = ctx.adminApi().instance(IUser.class, domainUid).getComplete(userUid);
		} catch (Exception e) {
			// Ignore error, fail below
		}

		if (user == null) {
			throw new CliException("User UID:" + userUid + " not found in domain UID: " + domainUid);
		}

		return new UserDomainUid(domainUid, user);
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}
}
