/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.user;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import net.bluemind.addressbook.api.AddressBookDescriptor;
import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.addressbook.api.IAddressBooksMgmt;
import net.bluemind.addressbook.api.IVCardService;
import net.bluemind.calendar.api.CalendarDescriptor;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.calendar.api.ICalendarsMgmt;
import net.bluemind.calendar.api.IVEvent;
import net.bluemind.cli.adm.RepairCommand;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.directory.common.SingleOrDomainOperation;
import net.bluemind.cli.sds.RestoreSdsMappingCommand;
import net.bluemind.cli.utils.Tasks;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.todolist.api.ITodoLists;
import net.bluemind.todolist.api.ITodoUids;
import net.bluemind.todolist.api.IVTodo;
import net.bluemind.utils.FileUtils;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "import", description = "import user data from an archive file, existing data will be erased.")
public class UserImportCommand extends SingleOrDomainOperation {

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("user");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return UserImportCommand.class;
		}
	}

	// seems zip can't find file sizes, 8192 is the default internal buffer anyway
	private static final int BUFFER_SIZE = 8192;

	@Option(names = "--archiveFile", required = true, description = "BM user archive path")
	public Path archiveFile = null;

	private String domainUid;

	@Override
	public void synchronousDirOperation(String domainUid, ItemValue<DirEntry> de) throws IOException {
		this.domainUid = domainUid;
		File archive = archiveFile.toFile();
		if (!archive.exists() || archive.isDirectory()) {
			throw new CliException("Invalid archive file");
		}

		Path tempDir = Files.createDirectory(Paths.get("bm-import"));
		try {
			extractArchive(archive.toPath(), tempDir);

			// skip first directory level
			importDatas(de, Files.list(tempDir).findFirst().get());

			ctx.info("Executing mailboxFilesystem repair");
			RepairCommand repairCommand = new RepairCommand();
			repairCommand.forTarget(de.value.email);
			repairCommand.forContext(ctx);
			repairCommand.ops = "mailboxFilesystem";
			repairCommand.run();

			ctx.info("Executing mailboxAcls repair");
			repairCommand.ops = "mailboxAcls";
			repairCommand.run();

		} catch (IOException e) {
			ctx.error("Error extracting archive " + e.getMessage());
			throw new CliException(e);
		} finally {
			FileUtils.delete(tempDir.toFile());
		}
	}

	private void extractArchive(Path archivePath, Path tempDir) throws IOException {
		ctx.info("Extracting {} to {} ", archivePath.toString(), tempDir);
		File tempDirFile = tempDir.toFile();

		try (FileInputStream fInS = new FileInputStream(archivePath.toString());
				ZipInputStream zipInS = new ZipInputStream(fInS)) {
			ZipEntry zipEntry;
			while ((zipEntry = zipInS.getNextEntry()) != null) {
				File fileEntry = new File(tempDirFile, zipEntry.getName());

				if (checkZipSlip(tempDirFile, fileEntry)) {
					throw new CliException(
							"Stopped before potential zip slip: file outside destination: " + zipEntry.getName());
				}

				if (zipEntry.isDirectory()) {
					Files.createDirectories(fileEntry.toPath());
				} else {
					byte[] buffer = new byte[BUFFER_SIZE];
					try (FileOutputStream fOutS = new FileOutputStream(fileEntry)) {
						int count = 0;
						while ((count = zipInS.read(buffer)) > 0) {
							fOutS.write(buffer, 0, count);
						}
					}
				}
				zipInS.closeEntry();
			}
			ctx.info("Archive file extracted successfully to " + tempDir.toString());
		} catch (Exception e) {
			ctx.info("error {} - {}", e.getMessage(), e.getStackTrace());
			throw e;
		}
	}

	private boolean checkZipSlip(File tempDirFile, File entryFile) throws IOException {
		return !entryFile.getCanonicalPath().startsWith(tempDirFile.getCanonicalPath() + File.separator);
	}

	private void importDatas(ItemValue<DirEntry> de, Path tempDir) throws IOException {
		Files.list(tempDir).forEach(s -> {
			try {
				Files.list(s).forEach(subData -> {
					try {
						switch (s.getName(s.getNameCount() - 1).toString()) {
						case "contact":
							importContacts(de, subData);
							break;
						case "calendar":
							importCalendars(de, subData);
							break;
						case "task":
							importTasks(de, subData);
							break;
						case "email":
							importMail(de, subData);
							break;
						case "notes":
							ctx.warn("Notes ignored: not supported for now.");
							break;
						default:
							ctx.error("Unknown data directory : " + s);
							break;
						}
					} catch (IOException e) {
						throw new CliException(e);
					}
				});
			} catch (Exception e) {
				throw new CliException("Error importing data", e);
			}
		});
	}

	private void importMail(ItemValue<DirEntry> de, Path dir) throws IOException {
		if (dir.toFile().isDirectory() || "index.json".equals(dir.getFileName().toString())) {
			return;
		}

		// restore sds currently does not handle restoring to another user
		if (!dir.getFileName().toString().startsWith(de.value.email)) {
			ctx.warn("Skipping mail import, importing to another user is not supported for now.");
			return;
		}

		ctx.info("Importing mails : " + dir.getFileName().toString());

		RestoreSdsMappingCommand restorator = new RestoreSdsMappingCommand();
		restorator.forContext(ctx);
		restorator.jsonFile = dir.toFile();
		restorator.run();
	}

	private void importCalendars(ItemValue<DirEntry> de, Path dir) throws IOException {
		Files.list(dir).forEach(cal -> importCalendar(de, cal));
	}

	private void importCalendar(ItemValue<DirEntry> de, Path icsFile) {
		String calName = decodeName(icsFile);
		ctx.info("Importing calendar : " + calName);

		// find calendar
		String calUid = null;
		if (calName.equals(de.displayName)) {
			calUid = ICalendarUids.defaultUserCalendar(de.uid);
		} else {
			// create a new one
			ICalendarsMgmt calMgmt = ctx.adminApi().instance(ICalendarsMgmt.class);
			CalendarDescriptor desc = new CalendarDescriptor();
			desc.domainUid = domainUid;
			desc.name = calName;
			desc.owner = de.uid;
			calUid = UUID.randomUUID().toString();
			calMgmt.create(calUid, desc);
		}

		TaskRef ref = ctx.adminApi().instance(IVEvent.class, calUid)
				.importIcs(cliUtils.getStreamFromFile(icsFile.toString()));
		Tasks.follow(ctx, false, ref,
				(de.value.email != null && !de.value.email.isEmpty()) ? (de.value.email + " (" + de.uid + ")") : de.uid,
				String.format("Fail to import calendar for entry %s", de));
	}

	private void importContacts(ItemValue<DirEntry> de, Path dir) throws IOException {
		Files.list(dir).forEach(cal -> importContact(de, cal));
	}

	private void importContact(ItemValue<DirEntry> de, Path vcfFile) {
		String abName = decodeName(vcfFile);
		ctx.info("Importing addressbook : {}", abName);

		try {
			String importedVcard = new String(Files.readAllBytes(vcfFile));
			if (importedVcard == null || importedVcard.isBlank()) {
				ctx.info("Nothing to import, skipping : {}", abName);
				return;
			}

			// find ab
			String abUid = null;
			if (abName.equals("Mes contacts")) {
				abUid = IAddressBookUids.defaultUserAddressbook(de.uid);
			} else if (abName.equals("Contacts collectés")) {
				abUid = IAddressBookUids.collectedContactsUserAddressbook(de.uid);
			} else {
				// create a new one
				IAddressBooksMgmt abMgmt = ctx.adminApi().instance(IAddressBooksMgmt.class, domainUid);
				AddressBookDescriptor desc = new AddressBookDescriptor();
				desc.domainUid = domainUid;
				desc.name = abName;
				desc.owner = de.uid;
				abUid = UUID.randomUUID().toString();
				abMgmt.create(abUid, desc, false);
			}

			TaskRef ref = ctx.adminApi().instance(IVCardService.class, abUid).importCards(importedVcard);
			Tasks.follow(ctx, false, ref,
					(de.value.email != null && !de.value.email.isEmpty()) ? (de.value.email + " (" + de.uid + ")")
							: de.uid,
					String.format("Fail to import addressbook for entry %s", de));
		} catch (Exception e) {
			throw new CliException("Error importing addressbook " + vcfFile.toString(), e);
		}
	}

	private void importTasks(ItemValue<DirEntry> de, Path dir) throws IOException {
		Files.list(dir).forEach(cal -> importTask(de, cal));
	}

	private void importTask(ItemValue<DirEntry> de, Path vcfFile) {
		String name = decodeName(vcfFile);
		ctx.info("Importing todolist : " + name);

		// find ab
		String uid = null;
		if (name.equals("Mes tâches")) {
			uid = ITodoUids.defaultUserTodoList(de.uid);
		} else {
			// create a new one
			ITodoLists mgmt = ctx.adminApi().instance(ITodoLists.class, domainUid);
			ContainerDescriptor desc = new ContainerDescriptor();
			desc.domainUid = domainUid;
			desc.name = name;
			desc.owner = de.uid;
			uid = UUID.randomUUID().toString();
			mgmt.create(uid, desc);
		}

		try {
			TaskRef ref = ctx.adminApi().instance(IVTodo.class, uid).importIcs(new String(Files.readAllBytes(vcfFile)));
			Tasks.follow(ctx, false, ref,
					(de.value.email != null && !de.value.email.isEmpty()) ? (de.value.email + " (" + de.uid + ")")
							: de.uid,
					String.format("Fail to import todo for entry %s", de));
		} catch (Exception e) {
			throw new CliException("Error importing todolist " + vcfFile.toString(), e);
		}
	}

	private String decodeName(Path file) {
		return cliUtils
				.decodeFilename(file.getFileName().toString().substring(0, file.getFileName().toString().length() - 4));
	}

	@Override
	public Kind[] getDirEntryKind() {
		return new Kind[] { Kind.USER };
	}
}