/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.user;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import com.github.freva.asciitable.AsciiTable;
import com.github.freva.asciitable.Column;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.system.api.Credential;
import net.bluemind.system.api.ICredentials;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;
import picocli.CommandLine.ArgGroup;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "credentials", description = "Manage user credentials")
public class UserCredentialsCommand implements ICmdLet, Runnable {

	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("user");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return UserCredentialsCommand.class;
		}
	}

	private CliContext ctx;
	private CliUtils cliUtils;

	@ArgGroup(exclusive = true, multiplicity = "1", heading = "User options%n")
	public UserOpts userOpts;

	public static class UserOpts {
		@Option(names = "--email", description = "User email%n")
		public String email;

		@ArgGroup(exclusive = false, heading = "User and domain UIDs%n")
		public UserUids uids;
	}

	public static class UserUids {
		@Option(names = "--user-uid", required = true, description = "User UID%n")
		public String userUid;

		@Option(names = "--domain-uid", required = true, description = "User domain UID%n")
		public String domainUid;
	}

	@ArgGroup(exclusive = true, multiplicity = "1", heading = "Credential options%n")
	public CredentialOpts credentialsOpts;

	public static class CredentialOpts {
		@Option(names = "--list", description = "List user credentials%n")
		public boolean listOpts = false;

		@ArgGroup(exclusive = false, heading = "Delete user credential%n")
		public CredentialDelete deleteOpts;
	}

	public static class CredentialDelete {
		@Option(names = "--delete", required = true, description = "Delete user credential%n")
		public boolean delete = false;

		@Option(names = "--credential-id", required = true, description = "Credential id to delete%n")
		public String crendentialId;
	}

	@Option(names = "--json", required = false, defaultValue = "false", description = "Display credentials using Json format\nTable format otherwise")
	public boolean json;

	@Override
	public void run() {
		UserUids userUids = getUserUid();

		if (credentialsOpts.listOpts) {
			listUserCredentials(userUids);
			return;
		}

		if (credentialsOpts.deleteOpts.delete) {
			deleteUserCredential(userUids);
		}
	}

	private void deleteUserCredential(UserUids userUids) {
		ctx.adminApi().instance(ICredentials.class, userUids.domainUid).removeUserCredential(userUids.userUid,
				credentialsOpts.deleteOpts.crendentialId);
	}

	private void listUserCredentials(UserUids userUids) {
		List<Credential> userCredentials = ctx.adminApi().instance(ICredentials.class, userUids.domainUid)
				.getObfuscatedUserCredentials(userUids.userUid).values;

		ctx.info(json ? JsonUtils.asString(userCredentials)
				: AsciiTable.getTable(userCredentials,
						Arrays.asList(new Column().header("ID").with(userCredential -> userCredential.id),
								new Column().header("Type").with(userCredential -> userCredential.type),
								new Column().header("Label").with(userCredential -> userCredential.label),
								new Column().header("Created (" + cliUtils.localTz + ")")
										.with(userCredential -> cliUtils.epochToLocalDate(userCredential.created)))));
	}

	private UserUids getUserUid() {
		if (userOpts.uids != null) {
			ItemValue<User> user = null;
			try {
				user = ctx.adminApi().instance(IUser.class, userOpts.uids.domainUid).getComplete(userOpts.uids.userUid);
			} catch (Exception e) {
				// Ignore error, fail below
			}

			if (user == null) {
				throw new CliException(
						"User UID:" + userOpts.uids.userUid + " not found in domain UID: " + userOpts.uids.domainUid);
			}

			return userOpts.uids;
		}

		UserUids uids = new UserUids();
		uids.domainUid = cliUtils.getDomainUidByEmail(userOpts.email);
		uids.userUid = userOpts.email.equals("admin0@global.virt") ? "admin0_global.virt"
				: cliUtils.getUserUidByEmail(userOpts.email);
		return uids;
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

}
