/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.sds;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import io.netty.util.concurrent.DefaultThreadFactory;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.streams.ReadStream;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.utils.JsonStreams;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.sds.sync.api.ISdsSync;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.ArchiveKind;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.sysconf.helper.SysConfHelper;
import net.bluemind.utils.ProgressPrinter;

public class SdsObjectChecker {
	private final CliContext ctx;
	Map<String, Optional<ISdsSyncStore>> originStores;
	private final int workers;
	private final File tempDir;
	private final Path errorLogPath;

	public SdsObjectChecker(CliContext ctx, int workers, File tempDir) {
		this.ctx = ctx;
		this.workers = workers;
		this.originStores = loadAllStores();
		this.tempDir = tempDir;
		this.errorLogPath = tempDir.toPath().resolve("error.log");
	}

	private Map<String, Optional<ISdsSyncStore>> loadAllStores() {
		Map<String, Optional<ISdsSyncStore>> sdsStores = new HashMap<>();
		SystemConf sysConf = SysConfHelper.getPlainSystemConf(ctx.adminApi());

		for (ItemValue<Server> server : Topology.get().all(TagDescriptor.mail_imap.getTag())) {
			sdsStores.put(server.uid, new SdsStoreLoader().forSysconf(sysConf, server.uid));
		}
		return sdsStores;
	}

	public ArchiveKind archiveKind(SystemConf sysconf) {
		String archiveKind = Optional.ofNullable(sysconf.stringValue(SysConfKeys.archive_kind.name())).orElse("cyrus");
		if (archiveKind.isBlank() || archiveKind.equalsIgnoreCase("none")) {
			archiveKind = "cyrus";
		}
		return ArchiveKind.fromName(archiveKind);
	}

	public void check() {
		ISdsSync sdsSyncApi = ctx.infiniteRequestTimeoutAdminApi().instance(ISdsSync.class);
		AtomicLong lastIndex = new AtomicLong(0L);

		ProgressPrinter progress = new ProgressPrinter(sdsSyncApi.count(lastIndex.get()));
		ctx.info("to check: {}", progress);

		ArrayBlockingQueue<String> q = new ArrayBlockingQueue<>(workers);
		ReadStream<Buffer> reader = VertxStream.read(sdsSyncApi.sync(lastIndex.get()));
		reader.pause();

		try (ExecutorService pool = Executors.newFixedThreadPool(workers,
				new DefaultThreadFactory("cli-sds-check-email"));
				OutputStream errorLogStream = Files.newOutputStream(errorLogPath, StandardOpenOption.APPEND,
						StandardOpenOption.CREATE)) {
			CompletableFuture<Void> future = new JsonStreams(ctx).consume(reader, body -> {
				progress.add();
				String type = body.getString("type");
				if (!type.equals("BODYADD")) {
					return;
				}

				try {
					q.put(body.getString("key")); // block until a slot is free
				} catch (InterruptedException ie) {
					Thread.currentThread().interrupt();
				}

				pool.execute(() -> {
					String guid = body.getString("key");
					String serverUid = body.getString("srv");
					long index = body.getLong("index");
					lastIndex.accumulateAndGet(index, Math::max);
					q.remove(guid); // NOSONAR

					if (type.equals("BODYADD")) {
						check(guid, serverUid, errorLogStream);
					}
				});
				if (progress.shouldPrint()) {
					ctx.info("progress: {}", progress);
				}

			});
			errorLogStream.flush();
			try {
				future.orTimeout(16, TimeUnit.DAYS).join();
			} catch (Exception e) {
				ctx.error("unknown error {}", e.getMessage());
			} finally {
				ctx.info("Last index checked: {}", lastIndex.get());
			}
		} catch (IOException e1) {
			ctx.error("Unable to open errorLog: {}", e1);
		}
	}

	private void check(String guid, String serverUid, OutputStream errorLogStream) {
		originStores.getOrDefault(serverUid, Optional.empty()).ifPresentOrElse(originStore -> {
			File tempDownload;
			try {
				tempDownload = File.createTempFile("sds-check-" + guid, null, tempDir);
			} catch (IOException e) {
				ctx.error("Unable to create temporary file: {}", e);
				return;
			}
			try {
				SdsResponse response = originStore.download(GetRequest.of("", guid, tempDownload.toString()));
				if (response.error != null) {
					addFail(errorLogStream, guid, response.error.toString());
				}
			} catch (Exception e) {
				addFail(errorLogStream, guid, e.getMessage());
			} finally {
				tempDownload.delete(); // NOSONAR
			}
		}, () -> {
			throw new ServerFault("SdsStore not found for serverUid=" + serverUid);
		});
	}

	private synchronized void addFail(OutputStream errorLogStream, String guid, String error) {
		ctx.error("Download guid failed: {}: {}", guid, error);
		try {
			errorLogStream.write((guid + "\n").getBytes());
		} catch (IOException e) {
			ctx.error("Unable to write to log {}: {}", errorLogPath, e.getMessage());
		}
	}

}
