/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.sds;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.dto.ExistRequest;
import net.bluemind.sds.dto.ExistResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.sysconf.helper.SysConfHelper;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "check-list", description = "Check SDS objets availability")
public class SdsCheckListCommand implements ICmdLet, Runnable {
	private CliContext ctx;

	public SdsCheckListCommand() {
		// OK
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	@Option(names = "--workers", description = "run with X workers")
	public int workers = 4;

	@Parameters(paramLabel = "<file>", description = "A file with the body GUIDs to check, one per-line")
	public Path file = null;

	private final BlockingQueue<String> toCheck = new ArrayBlockingQueue<>(2 * workers);

	@Override
	public void run() {
		ctx.info("Checking emails...");
		if (file == null || !file.toFile().exists()) {
			ctx.error("{} does not exist", file);
			return;
		}

		SystemConf sysConf = SysConfHelper.getPlainSystemConf(ctx.adminApi());

		ISdsSyncStore store = new SdsStoreLoader()
				.forSysconf(sysConf, Topology.get().any(TagDescriptor.mail_imap.getTag()).uid).orElseThrow();

		AtomicBoolean produced = new AtomicBoolean(false);
		Thread.ofPlatform().name("prod").start(() -> {
			try {
				Files.lines(file).forEach(guid -> {
					try {
						boolean accepted = false;
						do {
							accepted = toCheck.offer(guid, 1, TimeUnit.MINUTES);
						} while (!accepted);
					} catch (InterruptedException e) {
						Thread.currentThread().interrupt();
						return;
					}
				});
				produced.set(true);
			} catch (IOException e) {
				throw new CliException(e);
			}
		});
		LongAdder checked = new LongAdder();
		LongAdder present = new LongAdder();
		LongAdder missing = new LongAdder();
		List<Thread> vts = new ArrayList<>(workers);
		for (int i = 0; i < workers; i++) {
			vts.add(Thread.ofPlatform().name("cons-" + i).start(() -> {
				try {
					while (true) {
						String guid = toCheck.poll(5, TimeUnit.SECONDS);
						if (guid == null && produced.get()) {
							return;
						} else if (guid != null) {
							boolean inSds = check(guid, store);
							checked.increment();
							if (inSds) {
								present.increment();
							} else {
								missing.increment();
								ctx.warn("{} is missing", guid);
							}
						}
					}
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					return;
				}
			}));
		}
		while (!produced.get()) {
			try {
				ctx.info("Checked: {}, present: {}, missing: {}", checked.sum(), present.sum(), missing.sum());
				Thread.sleep(5000);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				return;
			}
		}
		for (Thread t : vts) {
			try {
				ctx.info("Waiting for {} completion.", t);
				t.join();
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				return;
			}
		}
		ctx.info("Checked: {}, present: {}, missing: {}", checked.sum(), present.sum(), missing.sum());

	}

	private boolean check(String guid, ISdsSyncStore store) {
		ExistResponse resp = store.exists(ExistRequest.of(guid));
		return resp.exists;

	}

	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("sds");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return SdsCheckListCommand.class;
		}
	}
}