/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.imap.endpoint.exec;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.MessageProducer;
import io.vertx.core.net.NetSocket;
import io.vertx.core.streams.WriteStream;
import net.bluemind.imap.endpoint.ImapContext;
import net.bluemind.imap.endpoint.driver.FetchedItem;
import net.bluemind.imap.endpoint.driver.MailPart;

public class FetchedItemStream implements WriteStream<FetchedItem> {

	private static final Logger logger = LoggerFactory.getLogger(FetchedItemStream.class);
	private static final Pattern PARTIAL_PATTERN = Pattern.compile("<\\d+(.\\d+)>");

	private final NetSocket socket;
	private final List<MailPart> spec;
	private final String why;
	private final Pending pending;
	private final ImapContext imapCtx;

	private int writeCnt = 0;

	private volatile boolean ended;

	private boolean uidCommand = true;

	private static class Pending {

		private static final int INITIAL_SIZE = 15000;
		// bigger than classic 1500 MTU
		private static final int THRESHOLD = 9000;
		private volatile Buffer pendingChunks; // NOSONAR
		private final ImapContext ctx;

		Pending(ImapContext ctx) {
			this.pendingChunks = Buffer.buffer(INITIAL_SIZE);
			this.ctx = ctx;
		}

		public synchronized Future<Void> write(Buffer b) {
			int pendingLen = pendingChunks.length();
			int incLen = b.length();

			// pending is empty & buffer is suitable for sending directly
			if (pendingLen == 0 && incLen >= THRESHOLD) {
				return ctx.sender().write(b);
			}

			pendingChunks.appendBuffer(b);
			if (pendingLen + incLen >= THRESHOLD) {
				return flushTo(ctx.sender());
			} else {
				return Future.succeededFuture();
			}
		}

		public synchronized Future<Void> flush() {
			return flushTo(ctx.sender());
		}

		private Future<Void> flushTo(MessageProducer<Buffer> sender) {
			Buffer toFlush = pendingChunks;
			pendingChunks = Buffer.buffer(INITIAL_SIZE);
			return sender.write(toFlush);
		}
	}

	public FetchedItemStream(ImapContext ctx, String why, List<MailPart> fetchSpec, boolean uidCommand) {
		this.imapCtx = ctx;
		this.socket = ctx.socket();
		this.spec = fetchSpec;
		this.why = why;
		this.uidCommand = uidCommand;
		this.pending = new Pending(ctx);
	}

	@Override
	public String toString() {
		return "fetchStream{%s}".formatted(why);
	}

	@Override
	public WriteStream<FetchedItem> exceptionHandler(Handler<Throwable> handler) {
		imapCtx.childExceptionHandler(handler);
		return this;
	}

	@Override
	public Future<Void> write(FetchedItem data) {
		writeCnt++;
		return pending.write(toBuffer(data));
	}

	@Override
	public void write(FetchedItem data, Handler<AsyncResult<Void>> handler) {
		writeCnt++;
		pending.write(toBuffer(data)).andThen(handler);
	}

	@Override
	public void end(Handler<AsyncResult<Void>> handler) {
		if (!ended) {
			ended = true;
			logger.trace("ending fetch after {} write(s)", writeCnt);
			pending.flush().andThen(handler);
		}
	}

	@Override
	public WriteStream<FetchedItem> setWriteQueueMaxSize(int maxSize) {
		return this;
	}

	@Override
	public boolean writeQueueFull() {
		return socket.writeQueueFull();
	}

	@Override
	public WriteStream<FetchedItem> drainHandler(Handler<Void> handler) {
		socket.drainHandler(handler);
		return this;
	}

	private Buffer ascii(Buffer b, String s) {
		return b.appendBytes(s.getBytes(StandardCharsets.US_ASCII));
	}

	private static final byte[] FETCH_END = ")\r\n".getBytes(StandardCharsets.US_ASCII);

	private Buffer toBuffer(FetchedItem fetched) {
		Buffer b = Buffer.buffer(8192);
		String response = (uidCommand) ? "* " + fetched.seq + " FETCH (UID " + fetched.uid + " "
				: "* " + fetched.seq + " FETCH (";
		ascii(b, response);

		int i = 0;
		int size = spec.size();
		for (MailPart mp : spec) {
			String k = mp.toString();

			ByteBuf v = fetched.properties.get(k);

			i++;
			if (v != null) {
				String outName = new String(mp.outName());
				Matcher matcher = PARTIAL_PATTERN.matcher(outName);
				if (matcher.find()) {
					String partialLength = matcher.group(1);
					outName = outName.replace(partialLength, "");
					b.appendString(outName).appendBuffer(Buffer.buffer(v));
				} else {
					b.appendBytes(mp.outName()).appendBuffer(Buffer.buffer(v));
				}
				if (i < size) {
					b.appendString(" ");
				}
			} else if (k.equals("UID") && !uidCommand && v == null) {
				b.appendString("UID " + fetched.uid);
				if (i < size) {
					b.appendString(" ");
				}
			}
		}
		b.appendBytes(FETCH_END);
		return b;
	}

}