/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BoundStatement;

import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.RecipientKind;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class SortedRecordCqlStore extends CqlAbstractStore {

	private static final Logger logger = LoggerFactory.getLogger(SortedRecordCqlStore.class);
	private final Container mbox;
	private final Container subtree;
	private int delTtlSeconds;

	public SortedRecordCqlStore(CqlSession s, Container subtree, Container mbox) {
		super(s);
		this.mbox = mbox;
		this.subtree = subtree;
		this.delTtlSeconds = (int) TimeUnit.DAYS
				.toSeconds(SysConfHelper.fromSharedMap().integerValue(SysConfKeys.archive_days.name(), 7));
	}

	private record SortField(String subject, String sender, Integer size, Instant date) {

	}

	public void dropSortingArtifacts(Item it) {
		dropSortingArtifacts(it, null, null);
	}

	public void dropPreviousSortingArtifacts(Item it) {
		SortField sv = unique("""
				SELECT subject, sender, size, date
				FROM s_mailbox_record
				WHERE subtree_id=? AND container_id=? AND item_id=?
				""", r -> new SortField(r.getString(0), r.getString(1), (Integer) r.getObject(2), r.getInstant(3)),
				voidPop(), subtree.id, mbox.id, it.id);
		if (sv != null) {
			batch(//
					delByDate(it, sv), //
					delBySize(it, sv), //
					delBySubject(it, sv), //
					delBySender(it, sv)//
			);
		}
	}

	public void dropSortingArtifacts(Item it, MailboxRecord rec, Supplier<MessageBody> body) {
		SortField sv = unique("""
				SELECT subject, sender, size, date
				FROM s_mailbox_record
				WHERE subtree_id=? AND container_id=? AND item_id=?
				""", r -> new SortField(r.getString(0), r.getString(1), (Integer) r.getObject(2), r.getInstant(3)),
				voidPop(), subtree.id, mbox.id, it.id);

		if (sv == null) {
			logger.debug("id {} has no sort values", it.id);
			sv = of(it, rec, body);
			if (sv != null) {
				voidCql("""
						INSERT INTO s_deleted_mailbox_record
						(subtree_id, container_id, item_id, sender, subject, size, date)
						VALUES
						(?, ?, ?, ?, ?, ?, ?)
						USING TTL ?
						""", //
						subtree.id, mbox.id, it.id, sv.sender, sv.subject, sv.size, sv.date, delTtlSeconds);
			} else {
				voidCql("""
						DELETE FROM s_deleted_mailbox_record
						WHERE subtree_id=? AND container_id=? AND item_id=?
						""", subtree.id, mbox.id, it.id);
			}
			return;
		}
		batch(//
				delByDate(it, sv), //
				delBySize(it, sv), //
				delBySubject(it, sv), //
				delBySender(it, sv), //
				b("""
						INSERT INTO s_deleted_mailbox_record
						(subtree_id, container_id, item_id, sender, subject, size, date)
						VALUES
						(?, ?, ?, ?, ?, ?, ?)
						USING TTL ?
						""", //
						subtree.id, mbox.id, it.id, sv.sender, sv.subject, sv.size, sv.date, delTtlSeconds));

	}

	private BoundStatement delBySender(Item it, SortField sv) {
		return b(
				"DELETE FROM s_mailbox_record_by_sender_desc WHERE subtree_id=? AND container_id=? AND sender=? AND item_id=?",
				subtree.id, mbox.id, sv.sender, it.id);
	}

	private BoundStatement delBySubject(Item it, SortField sv) {
		return b(
				"DELETE FROM s_mailbox_record_by_subject_desc WHERE subtree_id=? AND container_id=? AND subject=? AND item_id=?",
				subtree.id, mbox.id, sv.subject, it.id);
	}

	private BoundStatement delBySize(Item it, SortField sv) {
		return b(
				"DELETE FROM s_mailbox_record_by_size_desc WHERE subtree_id=? AND container_id=? AND size=? AND item_id=?",
				subtree.id, mbox.id, sv.size, it.id);
	}

	private BoundStatement delByDate(Item it, SortField sv) {
		return b(
				"DELETE FROM s_mailbox_record_by_date_desc WHERE subtree_id=? AND container_id=? AND date=? AND item_id=?",
				subtree.id, mbox.id, sv.date, it.id);
	}

	private SortField of(Item it, MailboxRecord rec, Supplier<MessageBody> body) {
		if (rec == null || body == null) {
			return null;
		}
		MessageBody mb = body.get();

		if (mb == null) {
			logger.warn("Null body for {}, sorting support will be incomplete", it.id);
			return null;
		}
		String subject = SubjectCleaner.clean(mb.subject);
		String sender = mb.recipients.stream().filter(r -> r.kind == RecipientKind.Originator)
				.map(r -> Optional.ofNullable(r.dn).orElse(r.address)).findFirst().orElse("");
		return new SortField(subject, sender, mb.size, rec.internalDate.toInstant());
	}

	public void writeSortingArtifacts(Item it, MailboxRecord rec, Supplier<MessageBody> body) {
		SortField sf = of(it, rec, body);
		if (sf == null) {
			logger.warn("Null SortField for {}", it.id);
			return;
		}

		batch(
				// master sort values to know what to delete later on
				b("""
						INSERT INTO s_mailbox_record
						(subtree_id, container_id, item_id, sender, subject, size, date)
						VALUES
						(?, ?, ?, ?, ?, ?, ?)
						""", subtree.id, mbox.id, it.id, sf.sender, sf.subject, sf.size, sf.date), //
				b("""
						INSERT INTO s_mailbox_record_by_date_desc
						(subtree_id, container_id, item_id, flagged, unseen, date)
						VALUES
						(?, ?, ?, ?, ?, ?)
						""", subtree.id, mbox.id, it.id, rec.flagged(), rec.unseen(), sf.date), b("""
						INSERT INTO s_mailbox_record_by_subject_desc
						(subtree_id, container_id, item_id, flagged, unseen, subject)
						VALUES
						(?, ?, ?, ?, ?, ?)
						""", subtree.id, mbox.id, it.id, rec.flagged(), rec.unseen(), sf.subject), //
				b("""
						INSERT INTO s_mailbox_record_by_size_desc
						(subtree_id, container_id, item_id, flagged, unseen, size)
						VALUES
						(?, ?, ?, ?, ?, ?)
						""", subtree.id, mbox.id, it.id, rec.flagged(), rec.unseen(), sf.size), //
				b("""
						INSERT INTO s_mailbox_record_by_sender_desc
						(subtree_id, container_id, item_id, flagged, unseen, sender)
						VALUES
						(?, ?, ?, ?, ?, ?)
						""", subtree.id, mbox.id, it.id, rec.flagged(), rec.unseen(), sf.sender) //
		);

	}

}