/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.shaded.guava.common.collect.Lists;

import net.bluemind.backend.mail.api.FolderCounters;
import net.bluemind.backend.mail.replica.api.AppendTx;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.repository.IMailboxReplicaStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.repository.sequences.ISequenceStore;
import net.bluemind.repository.sequences.Sequences;

public class CqlMailboxReplicaStore extends CqlAbstractStore implements IMailboxReplicaStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlMailboxReplicaStore.class);
	private final Container subtree;
	private final ISequenceStore seqs;

	public CqlMailboxReplicaStore(CqlSession forKeyspace, Container subtreeContainer, ISequenceStore seqs) {
		super(forKeyspace);
		this.subtree = subtreeContainer;
		this.seqs = seqs;
	}

	private static final EntityPopulator<MailboxReplica> POP = (Row r, int i, MailboxReplica m) -> {
		m.name = r.getString(i++);
		m.fullName = r.getString(i++);
		m.parentUid = r.getString(i++);

		return i;
	};

	@Override
	public void create(Item item, MailboxReplica value) throws SQLException {
		applyCql("""
				INSERT INTO t_mailbox_replica
				(short_name, name, parent_uid, unique_id, container_id, item_id)
				VALUES
				(?, ?, ?, ?, ?, ?)
				""", value.name, value.fullName, value.parentUid, item.uid, subtree.id, item.id);
	}

	@Override
	public void update(Item item, MailboxReplica value) throws SQLException {
		create(item, value);
	}

	@Override
	public void delete(Item item) throws SQLException {
		applyCql("""
				DELETE FROM t_mailbox_replica
				WHERE container_id = ? AND item_id = ?
				""", subtree.id, item.id);
	}

	@Override
	public MailboxReplica get(Item item) {
		return unique("""
				SELECT
				short_name, name, parent_uid
				FROM t_mailbox_replica
				WHERE container_id = ? AND item_id = ?
				""", r -> {
			MailboxReplica mr = new MailboxReplica();
			mr.uidValidity = subtree.id;
			String sn = MailSequences.imapUid(item.id);
			mr.lastUid = seqs.curVal(sn);
			logger.debug("lastUid set to {} from {}", mr.lastUid, sn);
			return mr;
		}, POP, subtree.id, item.id);
	}

	@Override
	public void deleteAll() throws SQLException {
		voidCql("DELETE FROM t_mailbox_replica WHERE container_id = ?", subtree.id);
	}

	@Override
	public String byName(String name) throws SQLException {
		Long itemId = unique("""
				SELECT
				item_id
				FROM t_replica_by_name
				WHERE container_id = ? AND name = ?
				""", r -> r.getLong(0), voidPop(), subtree.id, name);
		if (itemId == null) {
			return null;
		}
		return unique("select uid from t_container_item where container_id=? and id=?", r -> r.getString(0), voidPop(),
				subtree.id, itemId);
	}

	@Override
	public AppendTx prepareAppend(int count, long mboxReplicaId) throws SQLException {
		long newLast = seqs.nextVals(MailSequences.imapUid(mboxReplicaId), count);
		AppendTx tx = new AppendTx();
		tx.imapUid = newLast + count;
		tx.internalStamp = System.currentTimeMillis();
		return tx;
	}

	private record TreeNode(long itemId, String uniqueId, String parentUid) {
	}

	@Override
	public List<Long> childrensOf(String folderUid) throws SQLException {
		List<TreeNode> nodes = map("""
				SELECT item_id, unique_id, parent_uid
				FROM t_mailbox_replica
				where container_id=?
				""", r -> new TreeNode(r.getLong(0), r.getString(1), r.getString(2)), voidPop(), subtree.id);
		List<Long> children = new ArrayList<>(nodes.size());
		pumpChildren(children, folderUid, nodes);
		logger.info("{} has {} children", folderUid, children.size());
		return children;
	}

	private void pumpChildren(List<Long> children, String folderUid, List<TreeNode> nodes) {
		List<TreeNode> directChildren = nodes.stream().filter(n -> folderUid.equals(n.parentUid())).toList();
		if (directChildren.isEmpty()) {
			return;
		}
		for (TreeNode tn : directChildren) {
			children.add(tn.itemId());
			pumpChildren(children, tn.uniqueId(), nodes);
		}
	}

	@Override
	public boolean exists(Item item) throws SQLException {
		Long iid = unique("SELECT item_id FROM t_mailbox_replica WHERE container_id=?", r -> r.getLong(0), voidPop(),
				subtree.id);
		return iid != null;
	}

	@Override
	public List<MailboxReplica> getMultiple(List<Item> items) throws SQLException {
		return items.stream().map(this::get).toList();
	}

	private record ItemIdAndRecordContainer(long itemId, String uniqueId) {
	}

	private record ContIdUid(long contId, String contUid) {
	}

	private record Count(long contId, int cnt) {

	}

	@Override
	public List<FolderCounters> counters(List<Long> folderItemIds) throws SQLException {
		List<FolderCounters> ret = new ArrayList<>(folderItemIds.size());
		for (List<Long> slice : Lists.partition(folderItemIds, 64)) {

			List<ItemIdAndRecordContainer> resolvedContainers = map(
					"SELECT item_id, unique_id FROM t_mailbox_replica WHERE container_id=? AND item_id IN ?",
					r -> new ItemIdAndRecordContainer(r.getLong(0), IMailReplicaUids.mboxRecords(r.getString(1))),
					voidPop(), subtree.id, slice);
			List<ContIdUid> contIds = map("SELECT id, uid from t_container WHERE uid IN ?",
					r -> new ContIdUid(r.getLong(0), r.getString(1)), voidPop(),
					resolvedContainers.stream().map(ir -> ir.uniqueId).toList());
			Map<String, Long> containerToId = contIds.stream()
					.collect(Collectors.toMap(ciu -> ciu.contUid, ciu -> ciu.contId));
			Map<Long, Integer> unseenByCont = map(
					"SELECT container_id, unseen FROM cnt_items_unseen_visible where container_id IN ?",
					r -> new Count(r.getLong(0), (int) r.getLong(1)), voidPop(),
					contIds.stream().map(c -> c.contId).toList()).stream()
					.collect(Collectors.toMap(c -> c.contId, c -> c.cnt));
			Map<Long, Integer> totalByCont = map("SELECT container_id, all FROM cnt_items_all where container_id IN ?",
					r -> new Count(r.getLong(0), (int) r.getLong(1)), voidPop(),
					contIds.stream().map(c -> c.contId).toList()).stream()
					.collect(Collectors.toMap(c -> c.contId, c -> c.cnt));
			resolvedContainers.stream().map(irc -> {
				FolderCounters fc = new FolderCounters();
				fc.itemId = irc.itemId;
				fc.recordsVersion = seqs.curVal(Sequences.itemVersions(irc.uniqueId));
				long contId = containerToId.get(irc.uniqueId);
				fc.unseenVisible = unseenByCont.get(contId);
				fc.total = totalByCont.get(contId);
				fc.totalVisible = fc.total;
				return fc;
			}).forEach(ret::add);
		}

		return ret;
	}

}
