/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.webappdata.impl;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.webappdata.api.WebAppData;

public class WebappDataBackupRepository {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/webappdata");
	private static Logger logger = LoggerFactory.getLogger(WebappDataBackupRepository.class);
	RepositoryBackupPath repositoryPath;

	public WebappDataBackupRepository(RepositoryBackupPath repositoryPath) {
		this.repositoryPath = repositoryPath;
	}

	public WebappDataBackupRepository(Path repositoryPath) {
		this(new RepositoryBackupPath(repositoryPath));
	}

	record BackupWebAppDataRecord(ContainerDescriptor cd, List<ItemValue<WebAppData>> webAppDatas) {

	}

	public List<String> writeDatas(List<BackupWebAppDataRecord> backupWebAppDatas) throws IOException {
		List<String> ret = new ArrayList<>();
		for (BackupWebAppDataRecord entryJson : backupWebAppDatas) {
			Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(entryJson.cd);
			JsonReadWrite.write(JsonUtils.writer(RestorableWebAppDatas.class), jsonPathToWrite,
					new RestorableWebAppDatas(entryJson.cd, entryJson.webAppDatas));
			ret.add(RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite));
		}
		return ret;
	}

	private List<RestorableWebAppDatas> readZipFilesForUserWebAppDatas(String domainUid, String ownerUid)
			throws Exception {
		Optional<Path> zipFilePath = repositoryPath.getZipFilePathForOwner(domainUid, ownerUid);
		if (zipFilePath.isEmpty()) {
			return Collections.emptyList();
		}
		return extractUserWebAppDatasFromZip(ownerUid, zipFilePath.get());
	}

	private List<RestorableWebAppDatas> readJsonFilesForUserWebAppDatas(String ownerUid) throws IOException {
		Path restorablesPath = repositoryPath.resolveUserPath(ownerUid);
		return (List<RestorableWebAppDatas>) JsonReadWrite.readJsonFile(JsonUtils.reader(RestorableWebAppDatas.class),
				restorablesPath);
	}

	/**
	 * Returns the list of webappdata available in the backup
	 *
	 * @param ownerUid
	 * @return List of RestorableWebAppDatas
	 * @throws IOException
	 */
	public List<RestorableWebAppDatas> getRestorableWebappDatas(String domainUid, String ownerUid) throws Exception {
		List<RestorableWebAppDatas> restoredWebAppDatas = readZipFilesForUserWebAppDatas(domainUid, ownerUid);
		if (restoredWebAppDatas.isEmpty()) {
			restoredWebAppDatas = readJsonFilesForUserWebAppDatas(ownerUid);
		}

		if (restoredWebAppDatas.isEmpty()) {
			logger.warn("No webappdatas to restore for {}", ownerUid);
		}

		return restoredWebAppDatas;
	}

	private List<RestorableWebAppDatas> extractUserWebAppDatasFromZip(String folderName, Path zipFilePath) {
		List<RestorableWebAppDatas> webAppDatas = new ArrayList<>();
		try {
			JsonReadWrite.findJsonFilesWithPattern(folderName, zipFilePath).forEach(byteArray -> {
				try {
					webAppDatas.add((RestorableWebAppDatas) JsonReadWrite
							.readByteArr(JsonUtils.reader(RestorableWebAppDatas.class), byteArray));
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			});
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return webAppDatas;
	}

}
