/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.webappdata.impl;

import java.nio.file.Paths;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DPError.DPKind;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.RestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupPath;
import net.bluemind.dataprotect.webappdata.pg.RestoreWebAppDataTaskPg;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.webappdata.api.IWebAppData;
import net.bluemind.webappdata.api.IWebAppDataUids;

public class RestoreWebAppDataTask extends BlockingServerTask implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreWebAppDataTask.class);

	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;

	public RestoreWebAppDataTask(DataProtectGeneration backup, Restorable item) {
		this.backup = backup;
		this.restorableItem = new RestoreRestorableItem(item);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPKind.WEBAPPDATA.name()));

		WebappDataBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreWebAppDataTaskPg(backup, restorableItem).run();
		}
	}

	protected WebappDataBackupRepository getRepository() {
		try {
			List<PartGeneration> parts = backup.parts;
			PartGeneration corepart = parts.stream().filter(p -> WorkerDataType.WEBAPPDATA == p.getWorkerDataType())
					.findFirst().orElseThrow(() -> ServerFault.notFound("Unable to find backup part 'webappdata'"));
			ItemValue<Server> coreServer = Topology.get().core();
			RepositoryBackupPath backupPath = new RepositoryBackupPath(
					Paths.get(BackupPath.get(coreServer, TagDescriptor.bm_core.getTag()), String.valueOf(corepart.id),
							"var/backups/bluemind/webappdata"));
			if (backupPath.exists()) {
				return new WebappDataBackupRepository(backupPath);
			}
		} catch (Exception e) {
			logger.info(
					"Unable to get webappdata backup repository: {}, this message can be ignored if the restore process is related to a legacy database backup",
					e.getMessage());
		}
		return null;
	}

	private void restore(WebappDataBackupRepository repository) {
		BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();

		try {
			IWebAppData liveApi = live.provider().instance(IWebAppData.class,
					IWebAppDataUids.containerUid(restorableItem.liveEntryUid()));
			liveApi.deleteAll();

			for (var restorable : repository.getRestorableWebappDatas(restorableItem.domain(),
					restorableItem.liveEntryUid())) {
				if (logger.isInfoEnabled()) {
					logger.info("Backup contains {} webappdata", restorable.getItems().size());
				}
				restorableItem.monitorLog("Restoring webappdata container for user {}", restorableItem.displayName());

				restorable.getItems().forEach(item -> {
					liveApi.create(item.uid, item.value);
					restorableItem.monitorLog("Restoring webappdata '{}'", item.uid);
				});
			}
		} catch (Exception e) {
			logger.error("Error while restoring webappdata", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.WEBAPPDATA));
		} finally {
			restorableItem.endTask();
		}
	}
}
