/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.sds.store.s3.zstd;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.luben.zstd.RecyclingBufferPool;
import com.github.luben.zstd.ZstdOutputStream;
import com.netflix.spectator.api.DistributionSummary;

import io.netty.buffer.ByteBuf;
import net.bluemind.memory.pool.api.CommonMemoryPool;
import net.bluemind.memory.pool.api.IWritableChunk;
import net.bluemind.sds.store.PathHandler;
import software.amazon.awssdk.core.async.AsyncRequestBody;

public class ZstdRequestBody implements AsyncRequestBody {

	private static final Logger logger = LoggerFactory.getLogger(ZstdRequestBody.class);
	private IOException savedError;
	private ByteBuf mmap;
	private long len;
	private IWritableChunk chunk;

	public ZstdRequestBody(PathHandler sourceFile, DistributionSummary compressionRatio) {
		Path tmpPath = null;
		try (InputStream in = sourceFile.read()) {
			long origLen = sourceFile.size();
			this.chunk = CommonMemoryPool.getDefault().allocateEmpty((int) (1.25 * origLen));

			try (OutputStream out = chunk.appendStream();
					ZstdOutputStream zos = new ZstdOutputStream(out, RecyclingBufferPool.INSTANCE, -3)) {
				in.transferTo(zos);
			}
			this.mmap = chunk.openBuffer();
			this.len = chunk.getSize();
			long compressionPercent = origLen != 0 ? len * 100 / origLen : 100;
			compressionRatio.record(Math.max(0, 100 - compressionPercent));
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
			this.savedError = e;
		} finally {
			if (tmpPath != null) {
				try {
					Files.deleteIfExists(tmpPath);
				} catch (IOException e) {
					// OK
				}
			}
		}
	}

	@Override
	public void subscribe(Subscriber<? super ByteBuffer> s) {
		Subscription sub = new Subscription() {

			private ByteBuf mmapView = mmap.duplicate();

			private boolean cancelled;

			@Override
			public void request(long n) {
				if (savedError != null) {
					s.onError(savedError);
					return;
				}
				for (int i = 0; i < n && !cancelled; i++) {
					int remain = mmapView.readableBytes();
					int toGrab = Math.min(32768, remain);
					if (toGrab == 0) {
						chunk.release();
						s.onComplete();
						break;
					} else {
						byte[] tgt = new byte[toGrab];
						mmapView.readBytes(tgt);
						s.onNext(ByteBuffer.wrap(tgt));
					}
				}

			}

			@Override
			public void cancel() {
				this.cancelled = true;
				chunk.release();
			}

		};
		s.onSubscribe(sub);

	}

	@Override
	public Optional<Long> contentLength() {
		return Optional.of(len);
	}

}
