/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License)
 * or the CeCILL as published by CeCILL.info (version 2 of the License).
 *
 * There are special exceptions to the terms and conditions of the
 * licenses as they are applied to this program. See LICENSE.txt in
 * the directory of this program distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.tags;

import java.util.List;
import java.util.Optional;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.Tag;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "delete", description = "deletes a tag")
public class TagDeleteCommand implements Runnable, ICmdLet {
	private CliContext ctx;

	@Option(names = "--mailbox-uid", required = true, description = "the mailbox uid")
	String mailboxUid;

	@Option(names = "--domain", required = true, description = "the domain")
	public String domainUid;

	@Option(names = "--tag-name", required = true, description = "the tag name")
	public String tagName;

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("tags");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return TagDeleteCommand.class;
		}
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	@Override
	public void run() {

		ItemValue<Domain> domain = ctx.adminApi().instance(IDomains.class).findByNameOrAliases(domainUid);
		if (domain == null) {
			ctx.error("domain '" + domainUid + "' cannot be found");
			throw new CliException("domain '" + domainUid + "' cannot be found");
		}

		IDirectory admDirApi = ctx.adminApi().instance(IDirectory.class, domain.uid);
		List<DirEntry> dirEntries = admDirApi.getEntries(domain.uid);

		DirEntry dirEntry = dirEntries.stream()
				.filter(d -> d.kind.equals(BaseDirEntry.Kind.MAILSHARE) | d.kind.equals(BaseDirEntry.Kind.USER))
				.filter(d -> d.entryUid.equals(mailboxUid)).findFirst().orElse(null);

		if (dirEntry == null) {
			ctx.error("Cannot find mailbox '{}'", mailboxUid);
		}

		String containerTags = ITagUids.defaultTags(dirEntry.entryUid);

		ITags service = ctx.adminApi().instance(ITags.class, containerTags);

		ItemValue<Tag> itemValueTag = service.all().stream().filter(t -> t.value.label.equals(tagName)).findFirst()
				.orElse(null);
		if (itemValueTag == null) {
			ctx.error("'{}' - Tag '{}' does not exist", mailboxUid, tagName);
			return;
		}

		service.delete(itemValueTag.uid);
		ctx.info("'{}' - Tag '{}' deleted", mailboxUid, tagName);

	}
}
