import Vuex from "vuex";

import { cloneDeep } from "@bluemind/commons/utils/lang";
import ServiceLocator, { inject } from "@bluemind/inject";
import { MockMailboxItemsClient } from "@bluemind/test-utils";

import { default as storeOptions } from "~/store/messages";

import {
    alternativewithOneRelative,
    alternativewithTwoAttachments,
    alternativewithTwoAttachmentsOneInline,
    htmlContentWithInline,
    simpleAlternative,
    simpleDiv
} from "./data";

const key = "key";
const folderRef = { uid: "uid" };
let service, store, message;

describe("SET_MESSAGE_CONTENT", () => {
    beforeEach(() => {
        ServiceLocator.register({ provide: "MailboxItemsPersistence", use: new MockMailboxItemsClient() });
        service = inject("MailboxItemsPersistence", "uid");
        store = new Vuex.Store(cloneDeep(storeOptions));
        message = {
            key,
            folderRef: folderRef,
            structure: null
        };
    });
    test("add new inline upload a new part", async () => {
        const htmlContent = htmlContentWithInline;
        message.structure = simpleAlternative;

        store.commit("ADD_MESSAGES", { messages: [message] });
        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });
        expect(service.uploadPart).toHaveBeenCalled();
    });
    test("set message preview", async () => {
        const content = "content";
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });
        expect(store.state[key].preview).toBeFalsy();

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content });
        expect(store.state[key].preview).toEqual(content);
    });
    test("add new inline image in structure without attachments", async () => {
        const htmlContent = htmlContentWithInline;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });
        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });
        expect(store.state[message.key].structure).toEqual(alternativewithOneRelative);
    });
    test("add new inline image in structure with attachments", async () => {
        const htmlContent = htmlContentWithInline;
        message.structure = alternativewithTwoAttachments;
        store.commit("ADD_MESSAGES", { messages: [message] });
        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });
        expect(store.state[message.key].structure).toEqual(alternativewithTwoAttachmentsOneInline);
    });
    test("remove inline image in structure", async () => {
        const htmlContent = simpleDiv;
        message.structure = alternativewithOneRelative;
        store.commit("ADD_MESSAGES", { messages: [message] });
        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });
        expect(store.state[message.key].structure).toMatchObject(simpleAlternative);
    });
    test("should handle multiple inline images", async () => {
        const htmlContent = `
            <div>
                <img src="data:image/png;base64,abc123" data-bm-cid="<cid1>" alt="image1"/>
                <img src="data:image/jpeg;base64,def456" data-bm-cid="<cid2>" alt="image2"/>
            </div>
        `;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });
        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(service.uploadPart).toHaveBeenCalledTimes(4); // 2 images + 1 html + 1 text
    });

    test("should handle HTML content", async () => {
        const htmlContent = "<div><p>Just text content</p></div>";
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        const structure = store.state[message.key].structure;
        expect(structure).toMatchInlineSnapshot(`
          {
            "children": [
              {
                "address": "tmpAddress",
                "charset": "utf-8",
                "encoding": "quoted-printable",
                "mime": "text/plain",
              },
              {
                "address": "tmpAddress",
                "charset": "utf-8",
                "encoding": "quoted-printable",
                "mime": "text/html",
              },
            ],
            "mime": "multipart/alternative",
          }
        `);
    });

    test("should convert HTML to text for preview", async () => {
        const htmlContent = "<div><strong>Bold</strong> and <em>italic</em> text</div>";
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        const preview = store.state[key].preview;
        expect(preview).toMatchInlineSnapshot(`"Bold and italic text"`);
    });

    test("should handle error during content upload", async () => {
        const htmlContent = simpleDiv;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        service.uploadPart.mockRejectedValueOnce(new Error("Upload failed"));

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(store.state[key].status).toBe("SAVE_ERROR");
    });

    test("should handle images with duplicate CIDs", async () => {
        const htmlContent = `
            <div>
                <img src="data:image/png;base64,abc123" data-bm-cid="<same-cid>" alt="image1"/>
                <img src="data:image/png;base64,xyz789" data-bm-cid="<same-cid>" alt="image2"/>
            </div>
        `;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(store.state[message.key].structure).toMatchInlineSnapshot(`
          {
            "children": [
              {
                "address": "tmpAddress",
                "charset": "utf-8",
                "encoding": "quoted-printable",
                "mime": "text/plain",
              },
              {
                "children": [
                  {
                    "address": "tmpAddress",
                    "charset": "utf-8",
                    "encoding": "quoted-printable",
                    "mime": "text/html",
                  },
                  {
                    "address": "tmpAddress",
                    "contentId": "<same-cid>",
                    "dispositionType": "INLINE",
                    "encoding": "base64",
                    "fileName": "image1",
                    "mime": "image/png",
                    "size": 4,
                  },
                ],
                "mime": "multipart/related",
              },
            ],
            "mime": "multipart/alternative",
          }
        `);
    });

    test("should sanitize base64 data with invalid characters", async () => {
        const htmlContent = `<div><img src="data:image/png;base64,abc%0A123def" data-cid="<cid1>"/></div>`;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(service.uploadPart).toHaveBeenCalled();
        const structure = store.state[message.key].structure;
        expect(structure).toBeTruthy();
    });

    test("should replace image src with cid reference", async () => {
        const htmlContent = `<div><img src="data:image/png;base64,abc123" data-bm-cid="<test-cid>"/></div>`;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(service.uploadPart).not.toHaveBeenCalledWith(expect.stringContaining("data:image"));
    });

    test("should ignore images without data-cid attribute", async () => {
        const htmlContent = `<div><img src="data:image/png;base64,abc123"/></div>`;
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(service.uploadPart).not.toHaveBeenCalledWith(expect.any(Blob));
    });

    test("should handle empty HTML content", async () => {
        const htmlContent = "";
        message.structure = simpleAlternative;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(store.state[key].preview).toBe("");
    });

    test("should preserve non-image multipart children", async () => {
        const htmlContent = simpleDiv;
        const structureWithCalendar = cloneDeep(simpleAlternative);
        structureWithCalendar.children.push({
            mime: "text/calendar",
            children: []
        });
        message.structure = structureWithCalendar;
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        const structure = store.state[message.key].structure;
        const calendarPart = structure.children.find(c => c.mime === "text/calendar");
        expect(calendarPart).toBeTruthy();
    });

    test("should maintain mixed multipart structure with attachments", async () => {
        const htmlContent = simpleDiv;
        message.structure = alternativewithTwoAttachments;
        console.log();
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        const structure = store.state[message.key].structure;
        expect(structure).toMatchInlineSnapshot(`
          {
            "children": [
              {
                "children": [
                  {
                    "address": "tmpAddress",
                    "charset": "utf-8",
                    "encoding": "quoted-printable",
                    "mime": "text/plain",
                  },
                  {
                    "address": "tmpAddress",
                    "charset": "utf-8",
                    "encoding": "quoted-printable",
                    "mime": "text/html",
                  },
                ],
                "mime": "multipart/alternative",
              },
              {
                "address": "tmpAddress-attachment",
                "dispositionType": "ATTACHMENT",
                "fileName": "one",
                "mime": "image/png",
                "size": 8,
              },
              {
                "address": "tmpAddress-attachment",
                "dispositionType": "ATTACHMENT",
                "fileName": "two",
                "mime": "image/png",
                "size": 8,
              },
            ],
            "mime": "multipart/mixed",
          }
        `);
    });
    test("should handle structure without alternative part", async () => {
        message.structure = {
            mime: "multipart/mixed",
            children: [{ mime: "text/html", address: "addr" }]
        };
        const htmlContent = "<div><p>Hello world</p></div>";
        store.commit("ADD_MESSAGES", { messages: [message] });

        await store.dispatch("SET_MESSAGE_CONTENT", { message, content: htmlContent });

        expect(store.state[message.key].structure).toMatchInlineSnapshot(`
          {
            "children": [
              {
                "address": "tmpAddress",
                "charset": "utf-8",
                "encoding": "quoted-printable",
                "mime": "text/plain",
              },
              {
                "address": "tmpAddress",
                "charset": "utf-8",
                "encoding": "quoted-printable",
                "mime": "text/html",
              },
            ],
            "mime": "multipart/alternative",
          }
        `);
    });
    afterEach(() => {
        vi.clearAllMocks();
    });
});
