import { MimeType } from "@bluemind/email";
import ServiceLocator from "@bluemind/inject";
import { messageUtils } from "@bluemind/mail";
import { MockMailboxItemsClient } from "@bluemind/test-utils";

import { MY_DRAFTS } from "~/getters";
import { SET_MESSAGES_STATUS } from "~/mutations";

import { saveAsap } from "../../actions/save";
import { expandGroupsAndRemoveDuplicates } from "../../actions/saveHelper";

const { MessageAdaptor, MessageStatus, createWithMetadata } = messageUtils;
vi.mock("../../../api/apiMessages");
let itemsService, draft, context, saveParams;

const draftInternalId = "draft-internal-id";
const draftFolderKey = "draft-folder-key";
const messageCompose = { editorContent: "", inlineImagesSaved: [] };

global.structuredClone = val => JSON.parse(JSON.stringify(val));
ServiceLocator.register({ provide: "UserSession", factory: () => ({ bmBrandVersion: "TestVersion" }) });

const groupCMembers = [
    { dn: "Toto A", address: "toto@bm.net" },
    { dn: "Plop", address: "plop@bm.net" }
];
const groupDMembers = [
    { dn: "Plop", address: "plop@bm.net" },
    { dn: "Toto B", address: "toto@bm.net" }
];
jest.mock("@bluemind/contact", () => ({
    ...jest.requireActual("@bluemind/contact"),
    fetchMembersWithAddress: (containerUid, uid) => {
        return Promise.resolve(uid === "Group C" ? groupCMembers : uid === "Group D" ? groupDMembers : []);
    }
}));

describe("[Mail-WebappStore][actions] :  save", () => {
    beforeEach(() => {
        itemsService = new MockMailboxItemsClient();
        ServiceLocator.register({ provide: "MailboxItemsPersistence", factory: () => itemsService });

        draft = createWithMetadata({
            internalId: draftInternalId,
            folder: { key: draftFolderKey, uid: draftFolderKey }
        });
        draft.structure = {
            mime: "multipart/mixed",
            children: [
                {
                    mime: "multipart/alternative",
                    children: [
                        {
                            address: "old",
                            charset: "utf-8",
                            encoding: "quoted-printable",
                            mime: "text/plain"
                        },
                        {
                            address: "old",
                            charset: "utf-8",
                            encoding: "quoted-printable",
                            mime: "text/html"
                        }
                    ]
                },
                { address: "2" },
                { address: "3" }
            ]
        };
        draft.inlinePartsByCapabilities = [{ capabilities: [MimeType.TEXT_HTML], parts: [] }];
        draft.date = new Date();
        draft.status = MessageStatus.IDLE;

        saveParams = { draft, messageCompose, files: [] };
        MessageAdaptor.fromMailboxItem = vi.fn();
        MessageAdaptor.fromMailboxItem.mockReturnValue({ inlinePartsByCapabilities: null });
        context = {
            dispatch: vi.fn().mockReturnValue(Promise.resolve([1, 2, 3])),
            commit: vi.fn(),
            state: {
                [draft.key]: draft
            },
            rootGetters: {
                ["mail/" + MY_DRAFTS]: { key: draftFolderKey }
            }
        };
    });

    test("Save new draft", async () => {
        await saveAsap(context, saveParams);
        expect(context.commit).toHaveBeenNthCalledWith(1, SET_MESSAGES_STATUS, [
            {
                key: draft.key,
                status: MessageStatus.SAVING
            }
        ]);
        expect(context.commit).toHaveBeenNthCalledWith(8, SET_MESSAGES_STATUS, [
            {
                key: draft.key,
                status: MessageStatus.IDLE
            }
        ]);
        expect(itemsService.updateById).toHaveBeenCalledWith(draftInternalId, expect.anything());
    });

    test("Subject is modified", async () => {
        draft.subject = "Modified subject";
        await saveAsap(context, saveParams);

        expect(itemsService.updateById).toHaveBeenCalledWith(draftInternalId, expect.anything());
        const mailboxItemArg = itemsService.updateById.mock.calls[0][1];
        expect(mailboxItemArg).toMatchObject({ body: { subject: "Modified subject" } });
    });

    test("With error", async () => {
        itemsService.updateById.mockImplementation(() => {
            throw new Error();
        });
        await saveAsap(context, saveParams);
        expect(itemsService.updateById).toHaveBeenCalledWith(draftInternalId, expect.anything());
        expect(itemsService.updateById).toThrow(new Error());
        expect(context.commit).toHaveBeenNthCalledWith(5, SET_MESSAGES_STATUS, [
            {
                key: draft.key,
                status: MessageStatus.SAVE_ERROR
            }
        ]);
        vi.spyOn(global.console, "error");
    });
    test("No save when invalid draft structure", async () => {
        draft.structure = {
            mime: "text/plain",
            address: null
        };
        await saveAsap(context, saveParams);
        expect(itemsService.updateById).not.toHaveBeenCalled();
    });
    test("Expand groups and remove duplicated recipients", async () => {
        const groupCMembers = [
            { dn: "Toto A", address: "toto@bm.net" },
            { dn: "Plop", address: "plop@bm.net" }
        ];
        const groupDMembers = [
            { dn: "Plop", address: "plop@bm.net" },
            { dn: "Toto B", address: "toto@bm.net" }
        ];
        const recipients = [
            { dn: "Toto A", address: "toto@bm.net" },
            { dn: "Toto B", address: "toto@bm.net" },
            { dn: "Group A", address: "group@bm.net", kind: "group" },
            { dn: "Group B", address: "group@bm.net", kind: "group" },
            { dn: "Group C", uid: "Group C", kind: "group", members: groupCMembers },
            { dn: "Group B", uid: "Group D", kind: "group", members: groupDMembers }
        ];
        draft.to = [...recipients];
        draft.cc = [...recipients];
        draft.bcc = [...recipients];
        draft.replyTo = [...recipients];

        const results = await expandGroupsAndRemoveDuplicates(draft);

        const expectedEntry = [
            { dn: "Toto A", address: "toto@bm.net" },
            { dn: "Group A", address: "group@bm.net", kind: "group" },
            { dn: "Plop", address: "plop@bm.net" }
        ];
        const expected = { to: expectedEntry, cc: expectedEntry, bcc: expectedEntry, replyTo: expectedEntry };
        expect(results).toEqual(expected);
    });
});
