import { debounce } from "@bluemind/commons/utils/function";

const DEBOUNCE_TIME = 1000;

let resolve: (value?: unknown) => void | undefined;
let reject: (value?: unknown) => void | undefined;
const registeredCallback = new Map();
let promise: Promise<unknown> | undefined;

const waitingRoom = debounce(() => resolve && resolve(), DEBOUNCE_TIME);
export enum Actions {
    SET_CONTENT,
    SAVE,
    SEND
}
const ActionOrder = [Actions.SET_CONTENT, Actions.SAVE, Actions.SEND];
function executeActions(immediate = false) {
    if (!isInProgress()) {
        promise = new Promise((res, rej) => {
            resolve = res;
            reject = rej;
        });
        for (const type of ActionOrder) {
            promise = promise?.then(execute.bind(null, type));
        }
        promise = promise.then(clear).catch(err => {
            promise = undefined;
            if (err !== "CANCEL") {
                throw err;
            }
        });
    }
    waitingRoom();
    if (immediate) {
        waitingRoom.flush();
    }
    return promise;
}

function validate(type: Actions) {
    const blockingActions = ActionOrder.slice(0, ActionOrder.indexOf(type));
    if (!registeredCallback.has(type) || blockingActions?.some(action => registeredCallback.has(action))) {
        return false;
    }
    return true;
}

function execute(type: Actions, ...args: unknown[]) {
    if (validate(type)) {
        const callback = registeredCallback.get(type);
        registeredCallback.delete(type);
        return callback(...args);
    } else {
        return Promise.resolve();
    }
}

export async function scheduleAction(callback: (args: unknown) => Promise<unknown>, type: Actions, immediate = false) {
    registeredCallback.set(type, callback);
    return executeActions(immediate);
}

export async function flush() {
    return executeActions(true);
}

async function clear<T>(result: T): Promise<T> {
    promise = undefined;
    if (registeredCallback.size > 0) {
        executeActions();
    }
    return result;
}

export async function cancelSchedulerActions(checkCondition: () => boolean = () => true) {
    if (isInProgress() && checkCondition()) {
        registeredCallback.clear();
        waitingRoom.cancel();
        reject("CANCEL");
    }
    return;
}

function isInProgress() {
    return !!promise;
}
