import type { RouteLocation, RouteLocationRaw, RouteRecordRedirect } from "vue-router";

import ConversationPathParam from "../ConversationPathParam";
import MessagePathParam from "../MessagePathParam";
import MessageQueryParam from "../MessageQueryParam";

type VirtualRouteParams = Record<string, typeof String | typeof Array>;

type VirtualRouteLocation<T extends VirtualRouteParams> = Omit<RouteLocation, "params"> & {
    params: {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        [K in keyof T]: any;
    };
};

type VirtualRouteDeclaration<T extends VirtualRouteParams> = Omit<VirtualRoute<T>, "path">;

type VirtualRoute<T extends VirtualRouteParams> = Omit<RouteRecordRedirect, "redirect"> & {
    redirect: (to: VirtualRouteLocation<T>) => RouteLocationRaw;
};

const messageQueryPathParams: VirtualRouteParams = {
    mailbox: String,
    folder: String,
    search: String,
    filter: String,
    sort: String,
    messagequery: Array
};

const messagePathParams: VirtualRouteParams = {
    messagepath: String,
    message: String,
    action: String,
    related: String
};

const conversationPathParams: VirtualRouteParams = {
    conversation: String
};

function virtualRoutePath(params: VirtualRouteParams) {
    return (
        "/virtual/" +
        Object.keys(params)
            .map(param => (params[param] === Array ? `/:${param}*` : `:${param}?`))
            .join()
    );
}

function create<T extends VirtualRouteParams>(params: T, route: VirtualRouteDeclaration<T>): VirtualRoute<T> {
    return { ...route, path: virtualRoutePath(params) };
}

export default [
    create(
        { ...messageQueryPathParams, ...messagePathParams },
        {
            name: "v:mail:message",
            redirect(to) {
                const messagequery = MessageQueryParam.build(to.params.messagequery, to.params);
                const messagepath = MessagePathParam.build(to.params.messagepath, to.params.message);
                const params = messagequery ? { messagequery } : {};
                if (messagepath) {
                    return { name: "mail:message", params: { ...params, messagepath } };
                } else {
                    return { name: "mail:home", params };
                }
            }
        }
    ),
    create(
        { ...messageQueryPathParams, ...messagePathParams, ...conversationPathParams },
        {
            name: "v:mail:conversation",
            redirect(to) {
                const messagequery = MessageQueryParam.build(to.params.messagequery, to.params);
                const conversationpath = ConversationPathParam.build(
                    to.params.messagepath,
                    to.params.conversation,
                    to.params.action,
                    to.params.related
                );
                const params = messagequery ? { messagequery } : {};
                if (conversationpath) {
                    return {
                        name: "mail:conversation",
                        params: { ...params, conversationpath }
                    };
                } else {
                    return { name: "mail:home", params };
                }
            }
        }
    ),
    create(
        { ...messageQueryPathParams },
        {
            name: "v:mail:home",
            redirect(to) {
                const messagequery = MessageQueryParam.build(to.params.messagequery, to.params);
                return {
                    name: "mail:home",
                    params: messagequery ? { messagequery } : {}
                };
            }
        }
    )
];
