import { computed } from "vue";

import { useBmConfirm, useBmDynamicDialog } from "@bluemind/bluetiful";
import { createDefaultContact } from "@bluemind/business-components/src/Contact/autocompleteContact";
import { isEqual } from "@bluemind/commons/utils/lang";
import { capitalize } from "@bluemind/commons/utils/string";
import { RecipientAdaptor } from "@bluemind/contact";
import { EmailValidator, RECIPIENT_TYPES } from "@bluemind/email";
import i18n from "@bluemind/i18n";
import { mailTipUtils } from "@bluemind/mail";
import store from "@bluemind/store";

import { DEBOUNCED_SAVE_MESSAGE } from "~/actions";
import { useGetMailTipsCommand } from "~/commands";
import { SET_INVALID_RECIPIENT_ALERT } from "~/mutations";

import MailComposerInvalidRecipientDialog from "../components/RecipientPicker/MailComposerInvalidRecipientDialog.vue";

type Contact = {
    address?: string;
    containerUid?: string;
    dn?: string;
    isInternal?: boolean;
    kind?: "group" | "individual";
    memberCount?: number;
    photo?: boolean;
    uid?: string;
};

type Message = {
    subject: string;
    key: string;
    to: Contact[];
    cc: Contact[];
    from: Contact;
    bcc: Contact[];
    replyTo: Contact[];
};

type RecipientType = "to" | "cc" | "bcc" | "replyTo";

export function useValidateBeforeSend(message: Message) {
    const { getMailTipContext } = mailTipUtils;
    const execGetMailTips = useGetMailTipsCommand();
    const { open } = useBmDynamicDialog();
    const { confirm } = useBmConfirm();

    const messageCompose = computed(() => store.state.mail.messageCompose);

    function addNewContact(query: string, recipientType: RecipientType) {
        const currentContacts = RecipientAdaptor.toContacts(message[recipientType]);
        const newContact = createDefaultContact(query);
        return [...RecipientAdaptor.fromContacts(currentContacts), newContact];
    }

    function updateMessage(recipientType: RecipientType, updatedContacts: Contact[]) {
        store.commit(`mail/SET_MESSAGE_${recipientType.toUpperCase()}`, {
            messageKey: message.key,
            [recipientType]: updatedContacts
        });

        execGetMailTips({
            context: getMailTipContext(message),
            message: message
        });
        store.dispatch(DEBOUNCED_SAVE_MESSAGE, { draft: message });
    }

    async function validateRecipients() {
        Object.values(RECIPIENT_TYPES as RecipientType[]).forEach(recipientType => {
            const inputValue = messageCompose.value[`inputValue${capitalize(recipientType)}`];
            if (inputValue) {
                const updatedContacts = addNewContact(inputValue, recipientType);
                store.commit(`mail/SET_INPUT_VALUE_FIELD_${recipientType.toUpperCase()}`, "");
                if (!isEqual(message[recipientType], updatedContacts)) {
                    updateMessage(recipientType, updatedContacts);
                }
            }
        });

        const invalidContactsList = invalidContacts();

        if (invalidContactsList.length) {
            const sendAnyway = await open(MailComposerInvalidRecipientDialog, {
                data: { message }
            });
            if (!sendAnyway) {
                store.commit(`mail/${SET_INVALID_RECIPIENT_ALERT}`, true);
            }
            return sendAnyway;
        }
        return true;
    }

    function invalidContacts() {
        const getInvalidAddresses = (contacts: Contact[]) => {
            return contacts
                .filter(
                    contact =>
                        !EmailValidator.validateAddress(contact.address) && !(contact.kind?.toLowerCase() === "group")
                )
                .map(contact => contact.address);
        };

        return [
            ...getInvalidAddresses(message.to),
            ...getInvalidAddresses(message.cc),
            ...getInvalidAddresses(message.bcc),
            ...getInvalidAddresses(message.replyTo)
        ];
    }

    function confirmEmptySubject() {
        if (messageSubjectIsEmpty()) {
            return confirm({
                title: i18n.global.t("mail.compose.modal.empty_subject"),
                content: i18n.global.t("mail.compose.modal.empty_subject_message"),
                okLabel: i18n.global.t("common.send_anyway"),
                closeable: true
            });
        }
        return true;
    }

    function messageSubjectIsEmpty() {
        return !message.subject?.trim();
    }

    return {
        confirmEmptySubject,
        validateRecipients
    };
}
