import type { AlertMessage, AlertOptions } from "@bluemind/bluetiful";
import { toArray } from "@bluemind/commons/utils/array";
import store from "@bluemind/store";

enum Component {
    Composer = "COMPOSER",
    RecipientPicker = "RECIPIENTPICKER"
}
interface AlertsTracker {
    [Component.Composer]: Set<string>;
    [Component.RecipientPicker]: Set<string>;
}
type Message = { key: string };
type AlertTipMessage = AlertMessage & {
    renderer: {
        props: {
            message: Message;
        };
    };
};

const dispatched: Map<string, AlertsTracker> = new Map();
const closed: Map<string, Set<string>> = new Map();

export function useMailTipsAlerts() {
    function initMailTipsAlerts(messageKey: string) {
        dispatched.set(messageKey, {
            [Component.Composer]: new Set<string>(),
            [Component.RecipientPicker]: new Set<string>()
        });
        closed.set(messageKey, new Set<string>());
    }

    function clearMailtipsAlerts(messageKey: string) {
        dispatched.delete(messageKey);
        closed.delete(messageKey);
    }

    function isAlertTip(alert: AlertMessage): alert is AlertTipMessage {
        return !!alert.renderer.props?.message;
    }

    function alertRemoved(alert: AlertMessage | AlertMessage[]) {
        const alerts: AlertMessage[] = toArray(alert);
        alerts.forEach(a => {
            if (isAlertTip(a) && !a.uid.includes("MULTIPLE")) {
                const message = a.renderer.props?.message;
                closed.get(message.key)?.add(a.uid);
            }
        });
    }

    function alertWasClosed(uid: string, messageKey: string): boolean {
        return closed.get(messageKey)?.has(uid) || false;
    }

    function processMailTipsAlerts(
        newTips: { alert: AlertTipMessage; options: AlertOptions }[],
        prevTips: { alert: AlertTipMessage; options: AlertOptions }[],
        message: Message,
        component: Component
    ): { toRemove: Set<string>; toDispatch: { alert: AlertMessage; options: AlertOptions }[] } {
        const toRemove = new Set<string>(prevTips?.map(({ alert }) => alert.uid));
        const toDispatch: { alert: AlertTipMessage; options: AlertOptions }[] = [];
        newTips?.forEach(({ alert, options }) => {
            const uid = alert.uid;
            if (!toRemove.has(uid) && !alertWasClosed(uid, message.key)) {
                const alertAndMsg = { ...alert };
                alertAndMsg.renderer.props.message = message;
                toDispatch.push({ alert: alertAndMsg, options });
                dispatched.get(message.key)?.[component]?.add(uid);
            }
            toRemove.delete(uid);
        });
        return { toRemove, toDispatch };
    }

    function getDispatchedMailtipsAlerts(messageKey: string, component: Component): Set<string> {
        return dispatched.get(messageKey)?.[component] || new Set<string>();
    }

    function hasMailtipIcon(email: string): boolean {
        return (
            store.state.mail.messageCompose.overQuotaRecipients.includes(email) ||
            store.state.mail.messageCompose.outOfOfficeRecipients.includes(email)
        );
    }

    function mailTipIcon(email: string): string {
        return store.state.mail.messageCompose.overQuotaRecipients.includes(email) ? "inbox-full" : "magnetic-tape";
    }

    return {
        hasMailtipIcon,
        mailTipIcon,
        initMailTipsAlerts,
        processMailTipsAlerts,
        getDispatchedMailtipsAlerts,
        clearMailtipsAlerts,
        alertRemoved
    };
}
