import { markRaw } from "vue";

import { DefaultAlert } from "@bluemind/alert.store";
import { AlertTypes, useDispatchAlert } from "@bluemind/bluetiful";
import { MimeType } from "@bluemind/email";
import i18n from "@bluemind/i18n";
import { messageUtils, draftUtils } from "@bluemind/mail";
import router from "@bluemind/router";
import store from "@bluemind/store";

import { useAddAttachmentsCommand } from "~/commands/AddAttachmentsCommand";
import { SET_MESSAGE_SUBJECT, SET_MESSAGE_STRUCTURE } from "~/mutations";

import apiMessages from "~/store/api/apiMessages";

import { getIdentityForNewMessage, setFrom } from "../ComposerFrom";

import { buildBasicStructure } from "./initStructure";

const { MessageCreationModes, createEmlName } = messageUtils;
const { computeSubject } = draftUtils;
export default function useForwardEml() {
    const { execAddAttachments } = useAddAttachmentsCommand();

    const { dispatchAlert } = useDispatchAlert();

    async function initForwardEml(message, related) {
        const { message: relatedMessage } = related;

        const identity = getIdentityForNewMessage();
        await setFrom(identity, message);
        store.commit(`mail/${SET_MESSAGE_STRUCTURE}`, {
            messageKey: message.key,
            structure: buildBasicStructure()
        });

        const subject = computeSubject(MessageCreationModes.FORWARD, relatedMessage);
        store.commit(`mail/${SET_MESSAGE_SUBJECT}`, { messageKey: message.key, subject });
        try {
            const content = await apiMessages.fetchComplete(relatedMessage);
            const file = new File([content], createEmlName(relatedMessage, i18n.global.t("mail.viewer.no.subject")), {
                type: MimeType.EML
            });
            await execAddAttachments({ files: [file], message });
        } catch {
            dispatchAlert({
                uid: "FWD_EML_UID",
                renderer: {
                    component: markRaw(DefaultAlert),
                    props: {
                        name: "mail.forward_eml.fetch"
                    }
                },
                type: AlertTypes.ERROR
            });
            const conversation = store.state.mail.conversations.conversationByKey[relatedMessage.conversationRef.key];
            router.navigate({ name: "v:mail:conversation", params: { conversation } });
            return;
        }
        return message;
    }

    return { initForwardEml };
}
