import { encodeTag } from "@bluemind/commons/utils/tag";

import { ConversationListFilter } from "../../../store/conversationList";

import { escapeTerm } from "./escape";
import { QUERY_FIELDS, RECORD_QUERY_FIELDS } from "./Keywords";

const AND_SEPARATOR = " AND ";

export default function buildQuery(
    pattern: { [key: string]: string | string[] },
    filter: ConversationListFilter,
    folderUid: string,
    deep?: boolean
) {
    const { query, recordQuery } = splitFields(pattern);
    const completeRecordQuery = [...recordQuery, filterQuery(filter)].filter(Boolean).join(AND_SEPARATOR);
    const completeQuery = query.join(" ");

    return {
        query: completeQuery,
        recordQuery: completeRecordQuery,
        scope: searchScope(folderUid, deep)
    };
}
function splitFields(queryPattern: { [key: string]: string | string[] }): {
    query: string[];
    recordQuery: string[];
} {
    const query: string[] = [];
    const recordQuery: string[] = [];

    try {
        const queryFieldsValues = Object.values(QUERY_FIELDS);
        const recordQueryFieldsValues = Object.values(RECORD_QUERY_FIELDS);

        for (const [rawField, rawValue] of Object.entries(queryPattern)) {
            const { field, value } = normalize(rawField, rawValue);

            if (queryFieldsValues.includes(field)) {
                query.push(value);
            } else if (recordQueryFieldsValues.includes(field)) {
                recordQuery.push(value);
            }
        }

        return {
            recordQuery,
            query
        };
    } catch {
        return {
            recordQuery: [],
            query: [escapePattern(queryPattern)]
        };
    }
}

function normalize(rawField: string, rawValue: string | string[]) {
    const [field, value]: [string, string] =
        rawField === RECORD_QUERY_FIELDS.TAG
            ? [RECORD_QUERY_FIELDS.IS, encodeTagExpression(rawValue)]
            : [rawField, Array.isArray(rawValue) ? join(rawValue) : rawValue];

    const formattedValue = field === QUERY_FIELDS.CONTENT ? escapeAllSpecialCharacters(value) : `${field}:${value}`;
    return { field, value: formattedValue };
}

function join(arr: string[]) {
    if (arr.length === 1) {
        return arr.join();
    }
    return `(${arr.join(AND_SEPARATOR)})`;
}

function encodeTagExpression(rawValue: string | string[]): string {
    if (Array.isArray(rawValue)) {
        const encoded = rawValue.map(formatTag);
        return join(encoded);
    }
    return formatTag(rawValue);
}

function formatTag(tag: string) {
    return escapeTerm(encodeTag(tag));
}

function filterQuery(filter: ConversationListFilter): string {
    switch (filter) {
        case ConversationListFilter.UNREAD:
        case ConversationListFilter.FLAGGED:
            return "is:" + filter;
        default:
            return "";
    }
}

function searchScope(
    folderUid: string,
    deep = false
): {
    folderScope: { folderUid: string };
    isDeepTraversal: boolean;
} {
    return {
        folderScope: { folderUid },
        isDeepTraversal: deep
    };
}

function escapePattern(pattern: { [key: string]: string | string[] }): string {
    const parts: string[] = [];

    for (const [key, value] of Object.entries(pattern)) {
        if (Array.isArray(value)) {
            const escapedValues = value.map(escapeAllSpecialCharacters).join(" ");
            parts.push(`${key}:(${escapedValues})`);
        } else {
            parts.push(`${key}:${escapeAllSpecialCharacters(value)}`);
        }
    }

    return parts.join(" ");
}

const charsToEscape = ["\\", "+", "-", "&&", "||", "!", "(", ")", "{", "}", "[", "]", "^", '"', "~", "*", "?", ":"];
export function escapeAllSpecialCharacters(str: string): string {
    let escaped = str;
    for (const char of charsToEscape) {
        escaped = escaped.split(char).join("\\" + char);
    }
    return escaped;
}
