import { mapActions, mapGetters, mapMutations } from "vuex";

import { Verb } from "@bluemind/core.container.api";
import { inject } from "@bluemind/inject";
import { loadingStatusUtils, mailboxUtils } from "@bluemind/mail";

import { FETCH_FOLDERS, FETCH_MAILBOXES, LOAD_MAX_MESSAGE_SIZE, INIT_UNREAD_COUNTS } from "~/actions";
import {
    MAILBOXES_ARE_LOADED,
    MAILBOX_BY_NAME,
    MAILBOX_FOLDERS,
    MY_MAILBOX,
    MY_MAILBOX_FOLDERS,
    MY_INBOX,
    MAILBOXES
} from "~/getters";
import { ADD_MAILBOXES } from "~/mutations";

const { LoadingStatus } = loadingStatusUtils;
const { create, MailboxType } = mailboxUtils;

export default {
    data: () => ({
        isBeingUnmounted: false
    }),
    computed: {
        ...mapGetters("mail", {
            MAILBOX_BY_NAME,
            MAILBOXES_ARE_LOADED,
            MAILBOXES,
            MY_INBOX,
            MY_MAILBOX_FOLDERS,
            MY_MAILBOX
        })
    },
    methods: {
        ...mapActions("mail", { LOAD_MAX_MESSAGE_SIZE, FETCH_FOLDERS, FETCH_MAILBOXES, INIT_UNREAD_COUNTS }),
        ...mapMutations("mail", { ADD_MAILBOXES }),
        ...mapMutations("root-app", ["SET_APP_STATE"]),
        $_BootstrapMixin_initMyMailbox() {
            const { userId: owner, defaultEmail: address } = inject("UserSession");
            const myMailbox = create({ owner, dn: address, address, type: MailboxType.USER });
            myMailbox.verbs = [Verb.Manage, Verb.Write, Verb.Read, Verb.Visible];
            myMailbox.manageable = true;
            myMailbox.writable = true;
            myMailbox.readable = true;
            this.ADD_MAILBOXES([myMailbox]);
        },
        async $_BootstrapMixin_loadMailbox() {
            await this.FETCH_MAILBOXES();
            const mailbox = this.MAILBOX_BY_NAME(this.$store.state.mail.route.mailbox);
            await this.FETCH_FOLDERS({ mailbox });
            this.INIT_UNREAD_COUNTS({
                mailbox,
                folders: this.$store.getters[`mail/${MAILBOX_FOLDERS}`](mailbox)
            });
        },
        async $_BootstrapMixin_loadAllMailboxes() {
            if (!this.MAILBOXES_ARE_LOADED) {
                await this.FETCH_MAILBOXES();
            }
            await Promise.all(
                this.MAILBOXES.filter(mailbox => mailbox.loading === LoadingStatus.NOT_LOADED).map(async mailbox => {
                    await this.FETCH_FOLDERS({ mailbox });
                    this.INIT_UNREAD_COUNTS({
                        mailbox,
                        folders: this.$store.getters[`mail/${MAILBOX_FOLDERS}`](mailbox)
                    });
                })
            );
        }
    },
    async created() {
        try {
            if (!this.MY_MAILBOX) {
                this.$_BootstrapMixin_initMyMailbox();
            }
            if (this.$store.state.mail.route.mailbox) {
                await this.$_BootstrapMixin_loadMailbox();
            }
            await this.FETCH_FOLDERS({ mailbox: this.MY_MAILBOX });
            await this.LOAD_MAX_MESSAGE_SIZE(inject("UserSession").userId);
            this.INIT_UNREAD_COUNTS({
                mailbox: this.MY_MAILBOX,
                folders: this.$store.getters[`mail/${MAILBOX_FOLDERS}`](this.MY_MAILBOX)
            });
            await this.$_BootstrapMixin_loadAllMailboxes();
        } catch (error) {
            if (!this.isBeingUnmounted) {
                // eslint-disable-next-line no-console
                console.error("Error while bootstraping application... ", error);
                this.SET_APP_STATE("error");
            }
        }
    },
    beforeUnmount() {
        this.isBeingUnmounted = true;
    }
};
