/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.tag.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IEventBusAccessRule;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.tag.api.ITagUids;

public class TagEventBusAccessRule implements IEventBusAccessRule {

	private static final String BASE_ADDRESS = "tags.";
	private static final Logger logger = LoggerFactory.getLogger(TagEventBusAccessRule.class);

	@Override
	public boolean match(String path) {
		return path.startsWith(BASE_ADDRESS) && !"tags.changed".equals(path);
	}

	@Override
	public boolean authorize(BmContext context, String path) {
		String uid = path.substring(BASE_ADDRESS.length());

		try {
			return RBACManager.forContext(context)
					.forContainer(IMailboxAclUids.uidForMailbox(ITagUids.mailboxForUid(uid))).can(Verb.Visible.name());
		} catch (ServerFault e) {
			if (e.getCode() == ErrorCode.NOT_FOUND) {
				if (!uid.contains("admin0_global.virt")) {
					logger.info("Authorization on non-existing tag container {} requested", uid);
				}
				return false;
			} else {
				throw e;
			}
		}
	}
}