/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.sds.store;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;

import net.bluemind.common.io.Buffered;
import net.bluemind.memory.pool.api.ChunkFiler;
import net.bluemind.memory.pool.api.IWritableChunk;

public abstract class PathHandler {

	public abstract OutputStream openForWriting() throws IOException;

	public abstract SeekableByteChannel openWriteChannel() throws IOException;

	public abstract InputStream read() throws IOException;

	public abstract int size();

	public void cleanup() {

	}

	private static final class RealPathHandler extends PathHandler {

		private final Path path;

		public RealPathHandler(Path path) {
			this.path = path;
		}

		@Override
		public OutputStream openForWriting() throws IOException {
			return Buffered.output(Files.newOutputStream(path, StandardOpenOption.CREATE,
					StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE));
		}

		@Override
		public SeekableByteChannel openWriteChannel() throws IOException {
			System.err.println("openChannel");
			return Files.newByteChannel(path, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING,
					StandardOpenOption.WRITE);
		}

		@Override
		public InputStream read() throws IOException {
			return Files.newInputStream(path);
		}

		@Override
		public int size() {
			return (int) path.toFile().length();
		}

		@Override
		public void cleanup() {
			try {
				Files.deleteIfExists(path);
			} catch (IOException e) {
				// OK
			}
		}

	}

	private static final class PoolChunkHandler extends PathHandler {

		private final IWritableChunk chunk;

		public PoolChunkHandler(IWritableChunk chunk) {
			this.chunk = chunk;
		}

		@Override
		public OutputStream openForWriting() throws IOException {
			return chunk.appendStream();
		}

		@Override
		public SeekableByteChannel openWriteChannel() throws IOException {
			return new SeekableByteChannel() {
				volatile boolean closed = false;

				@Override
				public boolean isOpen() {
					return closed;
				}

				@Override
				public void close() throws IOException {
					closed = true;
				}

				@Override
				public int write(ByteBuffer src) throws IOException {
					int ret = src.remaining();
					chunk.append(src);
					return ret;
				}

				@Override
				public SeekableByteChannel truncate(long size) throws IOException {
					return this;
				}

				@Override
				public long size() throws IOException {
					return chunk.getSize();
				}

				@Override
				public int read(ByteBuffer dst) throws IOException {
					throw new UnsupportedOperationException();
				}

				@Override
				public SeekableByteChannel position(long newPosition) throws IOException {
					throw new UnsupportedOperationException();
				}

				@Override
				public long position() throws IOException {
					return chunk.getPosition();
				}
			};
		}

		@Override
		public InputStream read() throws IOException {
			return chunk.openStream();
		}

		@Override
		public int size() {
			return chunk.getSize();
		}

	}

	public static PathHandler forPathOrUri(String path) {
		if (path.startsWith(ChunkFiler.PREFIX)) {
			return new PoolChunkHandler(ChunkFiler.byAddress(path));
		} else {
			return new RealPathHandler(Paths.get(path));
		}
	}

}
