/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.control;

import com.datastax.oss.driver.api.core.AllNodesFailedException;
import com.datastax.oss.driver.api.core.AsyncAutoCloseable;
import com.datastax.oss.driver.api.core.auth.AuthenticationException;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverConfig;
import com.datastax.oss.driver.api.core.connection.ReconnectionPolicy;
import com.datastax.oss.driver.api.core.loadbalancing.NodeDistance;
import com.datastax.oss.driver.api.core.metadata.Node;
import com.datastax.oss.driver.api.core.metadata.NodeState;
import com.datastax.oss.driver.internal.core.channel.ChannelEvent;
import com.datastax.oss.driver.internal.core.channel.DriverChannel;
import com.datastax.oss.driver.internal.core.channel.DriverChannelOptions;
import com.datastax.oss.driver.internal.core.channel.EventCallback;
import com.datastax.oss.driver.internal.core.context.InternalDriverContext;
import com.datastax.oss.driver.internal.core.metadata.DistanceEvent;
import com.datastax.oss.driver.internal.core.metadata.NodeStateEvent;
import com.datastax.oss.driver.internal.core.metadata.TopologyEvent;
import com.datastax.oss.driver.internal.core.util.Loggers;
import com.datastax.oss.driver.internal.core.util.concurrent.CompletableFutures;
import com.datastax.oss.driver.internal.core.util.concurrent.Reconnection;
import com.datastax.oss.driver.internal.core.util.concurrent.RunOrSchedule;
import com.datastax.oss.driver.internal.core.util.concurrent.UncaughtExceptions;
import com.datastax.oss.driver.shaded.guava.common.collect.ImmutableList;
import com.datastax.oss.protocol.internal.Message;
import com.datastax.oss.protocol.internal.response.Event;
import com.datastax.oss.protocol.internal.response.event.SchemaChangeEvent;
import com.datastax.oss.protocol.internal.response.event.StatusChangeEvent;
import com.datastax.oss.protocol.internal.response.event.TopologyChangeEvent;
import edu.umd.cs.findbugs.annotations.NonNull;
import io.netty.util.concurrent.EventExecutor;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.WeakHashMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class ControlConnection
implements EventCallback,
AsyncAutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(ControlConnection.class);
    private final InternalDriverContext context;
    private final String logPrefix;
    private final EventExecutor adminExecutor;
    private final SingleThreaded singleThreaded;
    private volatile DriverChannel channel;

    public ControlConnection(InternalDriverContext context) {
        this.context = context;
        this.logPrefix = context.getSessionName();
        this.adminExecutor = context.getNettyOptions().adminEventExecutorGroup().next();
        this.singleThreaded = new SingleThreaded(context);
    }

    public CompletionStage<Void> init(boolean listenToClusterEvents, boolean reconnectOnFailure, boolean useInitialReconnectionSchedule) {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.init(listenToClusterEvents, reconnectOnFailure, useInitialReconnectionSchedule));
        return this.singleThreaded.initFuture;
    }

    public CompletionStage<Void> initFuture() {
        return this.singleThreaded.initFuture;
    }

    public boolean isInit() {
        return this.singleThreaded.initFuture.isDone();
    }

    public DriverChannel channel() {
        return this.channel;
    }

    public void reconnectNow() {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.reconnectNow());
    }

    @Override
    @NonNull
    public CompletionStage<Void> closeFuture() {
        return this.singleThreaded.closeFuture;
    }

    @Override
    @NonNull
    public CompletionStage<Void> closeAsync() {
        return this.forceCloseAsync();
    }

    @Override
    @NonNull
    public CompletionStage<Void> forceCloseAsync() {
        RunOrSchedule.on(this.adminExecutor, () -> this.singleThreaded.forceClose());
        return this.singleThreaded.closeFuture;
    }

    @Override
    public void onEvent(Message eventMessage) {
        if (!(eventMessage instanceof Event)) {
            LOG.warn("[{}] Unsupported event class: {}", (Object)this.logPrefix, (Object)eventMessage.getClass().getName());
        } else {
            LOG.debug("[{}] Processing incoming event {}", (Object)this.logPrefix, (Object)eventMessage);
            Event event = (Event)eventMessage;
            switch (event.type) {
                case "TOPOLOGY_CHANGE": {
                    this.processTopologyChange(event);
                    break;
                }
                case "STATUS_CHANGE": {
                    this.processStatusChange(event);
                    break;
                }
                case "SCHEMA_CHANGE": {
                    this.processSchemaChange(event);
                    break;
                }
                default: {
                    LOG.warn("[{}] Unsupported event type: {}", (Object)this.logPrefix, (Object)event.type);
                }
            }
        }
    }

    private void processTopologyChange(Event event) {
        TopologyChangeEvent tce = (TopologyChangeEvent)event;
        switch (tce.changeType) {
            case "NEW_NODE": {
                this.context.getEventBus().fire(TopologyEvent.suggestAdded(tce.address));
                break;
            }
            case "REMOVED_NODE": {
                this.context.getEventBus().fire(TopologyEvent.suggestRemoved(tce.address));
                break;
            }
            default: {
                LOG.warn("[{}] Unsupported topology change type: {}", (Object)this.logPrefix, (Object)tce.changeType);
            }
        }
    }

    private void processStatusChange(Event event) {
        StatusChangeEvent sce = (StatusChangeEvent)event;
        switch (sce.changeType) {
            case "UP": {
                this.context.getEventBus().fire(TopologyEvent.suggestUp(sce.address));
                break;
            }
            case "DOWN": {
                this.context.getEventBus().fire(TopologyEvent.suggestDown(sce.address));
                break;
            }
            default: {
                LOG.warn("[{}] Unsupported status change type: {}", (Object)this.logPrefix, (Object)sce.changeType);
            }
        }
    }

    private void processSchemaChange(Event event) {
        SchemaChangeEvent sce = (SchemaChangeEvent)event;
        this.context.getMetadataManager().refreshSchema(sce.keyspace, false, false).whenComplete((metadata, error) -> {
            if (error != null) {
                Loggers.warnWithException(LOG, "[{}] Unexpected error while refreshing schema for a SCHEMA_CHANGE event, keeping previous version", this.logPrefix, error);
            }
        });
    }

    private boolean isAuthFailure(Throwable error) {
        if (error instanceof AllNodesFailedException) {
            Collection<List<Throwable>> errors = ((AllNodesFailedException)error).getAllErrors().values();
            if (errors.size() == 0) {
                return false;
            }
            for (List<Throwable> nodeErrors : errors) {
                for (Throwable nodeError : nodeErrors) {
                    if (nodeError instanceof AuthenticationException) continue;
                    return false;
                }
            }
        }
        return true;
    }

    private static ImmutableList<String> buildEventTypes(boolean listenClusterEvents) {
        ImmutableList.Builder builder = ImmutableList.builder();
        builder.add("SCHEMA_CHANGE");
        if (listenClusterEvents) {
            ((ImmutableList.Builder)builder.add("STATUS_CHANGE")).add("TOPOLOGY_CHANGE");
        }
        return builder.build();
    }

    private class SingleThreaded {
        private final InternalDriverContext context;
        private final DriverConfig config;
        private final CompletableFuture<Void> initFuture = new CompletableFuture();
        private boolean initWasCalled;
        private final CompletableFuture<Void> closeFuture = new CompletableFuture();
        private boolean closeWasCalled;
        private final ReconnectionPolicy reconnectionPolicy;
        private final Reconnection reconnection;
        private DriverChannelOptions channelOptions;
        private final Map<Node, NodeDistance> lastNodeDistance = new WeakHashMap<Node, NodeDistance>();
        private final Map<Node, NodeState> lastNodeState = new WeakHashMap<Node, NodeState>();

        private SingleThreaded(InternalDriverContext context) {
            this.context = context;
            this.config = context.getConfig();
            this.reconnectionPolicy = context.getReconnectionPolicy();
            this.reconnection = new Reconnection(ControlConnection.this.logPrefix, ControlConnection.this.adminExecutor, () -> this.reconnectionPolicy.newControlConnectionSchedule(false), this::reconnect);
            CompletableFutures.whenCancelled(this.initFuture, () -> {
                LOG.debug("[{}] Init future was cancelled, stopping reconnection", (Object)ControlConnection.this.logPrefix);
                this.reconnection.stop();
            });
            context.getEventBus().register(DistanceEvent.class, RunOrSchedule.on(ControlConnection.this.adminExecutor, this::onDistanceEvent));
            context.getEventBus().register(NodeStateEvent.class, RunOrSchedule.on(ControlConnection.this.adminExecutor, this::onStateEvent));
        }

        private void init(boolean listenToClusterEvents, boolean reconnectOnFailure, boolean useInitialReconnectionSchedule) {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            if (this.initWasCalled) {
                return;
            }
            this.initWasCalled = true;
            try {
                ImmutableList eventTypes = ControlConnection.buildEventTypes(listenToClusterEvents);
                LOG.debug("[{}] Initializing with event types {}", (Object)ControlConnection.this.logPrefix, (Object)eventTypes);
                this.channelOptions = DriverChannelOptions.builder().withEvents(eventTypes, ControlConnection.this).withOwnerLogPrefix(ControlConnection.this.logPrefix + "|control").build();
                Queue<Node> nodes = this.context.getLoadBalancingPolicyWrapper().newControlReconnectionQueryPlan();
                this.connect(nodes, null, () -> this.initFuture.complete(null), error -> {
                    if (ControlConnection.this.isAuthFailure(error)) {
                        LOG.warn("[{}] Authentication errors encountered on all contact points. Please check your authentication configuration.", (Object)ControlConnection.this.logPrefix);
                    }
                    if (reconnectOnFailure && !this.closeWasCalled) {
                        this.reconnection.start(this.reconnectionPolicy.newControlConnectionSchedule(useInitialReconnectionSchedule));
                    } else {
                        if (error instanceof AllNodesFailedException) {
                            error = ((AllNodesFailedException)error).reword("Could not reach any contact point, make sure you've provided valid addresses");
                        }
                        this.initFuture.completeExceptionally((Throwable)error);
                    }
                });
            }
            catch (Throwable t) {
                this.initFuture.completeExceptionally(t);
            }
        }

        private CompletionStage<Boolean> reconnect() {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            Queue<Node> nodes = this.context.getLoadBalancingPolicyWrapper().newControlReconnectionQueryPlan();
            CompletableFuture<Boolean> result = new CompletableFuture<Boolean>();
            this.connect(nodes, null, () -> {
                result.complete(true);
                this.onSuccessfulReconnect();
            }, error -> result.complete(false));
            return result;
        }

        private void connect(Queue<Node> nodes, List<Map.Entry<Node, Throwable>> errors, Runnable onSuccess, Consumer<Throwable> onFailure) {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            Node node = nodes.poll();
            if (node == null) {
                onFailure.accept(AllNodesFailedException.fromErrors(errors));
            } else {
                LOG.debug("[{}] Trying to establish a connection to {}", (Object)ControlConnection.this.logPrefix, (Object)node);
                this.context.getChannelFactory().connect(node, this.channelOptions).whenCompleteAsync((channel, error) -> {
                    try {
                        NodeDistance lastDistance = this.lastNodeDistance.get(node);
                        NodeState lastState = this.lastNodeState.get(node);
                        if (error != null) {
                            if (this.closeWasCalled || this.initFuture.isCancelled()) {
                                onSuccess.run();
                            } else {
                                if (error instanceof AuthenticationException) {
                                    Loggers.warnWithException(LOG, "[{}] Authentication error", ControlConnection.this.logPrefix, error);
                                } else if (this.config.getDefaultProfile().getBoolean(DefaultDriverOption.CONNECTION_WARN_INIT_ERROR)) {
                                    Loggers.warnWithException(LOG, "[{}] Error connecting to {}, trying next node", ControlConnection.this.logPrefix, node, error);
                                } else {
                                    LOG.debug("[{}] Error connecting to {}, trying next node", new Object[]{ControlConnection.this.logPrefix, node, error});
                                }
                                List newErrors = errors == null ? new ArrayList() : errors;
                                newErrors.add(new AbstractMap.SimpleEntry<Node, Throwable>(node, (Throwable)error));
                                this.context.getEventBus().fire(ChannelEvent.controlConnectionFailed(node));
                                this.connect(nodes, newErrors, onSuccess, onFailure);
                            }
                        } else if (this.closeWasCalled || this.initFuture.isCancelled()) {
                            LOG.debug("[{}] New channel opened ({}) but the control connection was closed, closing it", (Object)ControlConnection.this.logPrefix, channel);
                            channel.forceClose();
                            onSuccess.run();
                        } else if (lastDistance == NodeDistance.IGNORED) {
                            LOG.debug("[{}] New channel opened ({}) but node became ignored, closing and trying next node", (Object)ControlConnection.this.logPrefix, channel);
                            channel.forceClose();
                            this.connect(nodes, errors, onSuccess, onFailure);
                        } else if (this.lastNodeState.containsKey(node) && (lastState == null || lastState == NodeState.FORCED_DOWN)) {
                            LOG.debug("[{}] New channel opened ({}) but node was removed or forced down, closing and trying next node", (Object)ControlConnection.this.logPrefix, channel);
                            channel.forceClose();
                            this.connect(nodes, errors, onSuccess, onFailure);
                        } else {
                            LOG.debug("[{}] New channel opened {}", (Object)ControlConnection.this.logPrefix, channel);
                            DriverChannel previousChannel = ControlConnection.this.channel;
                            ControlConnection.this.channel = channel;
                            if (previousChannel != null) {
                                LOG.debug("[{}] Forcefully closing previous channel {}", (Object)ControlConnection.this.logPrefix, (Object)previousChannel);
                                previousChannel.forceClose();
                            }
                            this.context.getEventBus().fire(ChannelEvent.channelOpened(node));
                            channel.closeFuture().addListener(f -> ControlConnection.this.adminExecutor.submit(() -> this.onChannelClosed((DriverChannel)channel, node)).addListener(UncaughtExceptions::log));
                            onSuccess.run();
                        }
                    }
                    catch (Exception e) {
                        Loggers.warnWithException(LOG, "[{}] Unexpected exception while processing channel init result", ControlConnection.this.logPrefix, e);
                    }
                }, (Executor)ControlConnection.this.adminExecutor);
            }
        }

        private void onSuccessfulReconnect() {
            boolean isFirstConnection = this.initFuture.complete(null);
            if (!isFirstConnection) {
                this.context.getMetadataManager().refreshNodes().whenComplete((result, error) -> {
                    if (error != null) {
                        LOG.debug("[{}] Error while refreshing node list", (Object)ControlConnection.this.logPrefix, error);
                    } else {
                        try {
                            this.context.getLoadBalancingPolicyWrapper().init();
                            this.context.getMetadataManager().refreshSchema(null, false, true).whenComplete((metadata, schemaError) -> {
                                if (schemaError != null) {
                                    Loggers.warnWithException(LOG, "[{}] Unexpected error while refreshing schema after a successful reconnection, keeping previous version", ControlConnection.this.logPrefix, schemaError);
                                }
                            });
                        }
                        catch (Throwable t) {
                            Loggers.warnWithException(LOG, "[{}] Unexpected error on control connection reconnect", ControlConnection.this.logPrefix, t);
                        }
                    }
                });
            }
        }

        private void onChannelClosed(DriverChannel channel, Node node) {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            if (!this.closeWasCalled) {
                this.context.getEventBus().fire(ChannelEvent.channelClosed(node));
                if (channel == ControlConnection.this.channel) {
                    LOG.debug("[{}] The current control channel {} was closed, scheduling reconnection", (Object)ControlConnection.this.logPrefix, (Object)channel);
                    this.reconnection.start();
                } else {
                    LOG.trace("[{}] A previous control channel {} was closed, reconnection not required", (Object)ControlConnection.this.logPrefix, (Object)channel);
                }
            }
        }

        private void reconnectNow() {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            if (this.initWasCalled && !this.closeWasCalled) {
                this.reconnection.reconnectNow(true);
            }
        }

        private void onDistanceEvent(DistanceEvent event) {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            this.lastNodeDistance.put(event.node, event.distance);
            if (event.distance == NodeDistance.IGNORED && ControlConnection.this.channel != null && !ControlConnection.this.channel.closeFuture().isDone() && event.node.getEndPoint().equals(ControlConnection.this.channel.getEndPoint())) {
                LOG.debug("[{}] Control node {} became IGNORED, reconnecting to a different node", (Object)ControlConnection.this.logPrefix, (Object)event.node);
                this.reconnectNow();
            }
        }

        private void onStateEvent(NodeStateEvent event) {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            this.lastNodeState.put(event.node, event.newState);
            if ((event.newState == null || event.newState == NodeState.FORCED_DOWN) && ControlConnection.this.channel != null && !ControlConnection.this.channel.closeFuture().isDone() && event.node.getEndPoint().equals(ControlConnection.this.channel.getEndPoint())) {
                LOG.debug("[{}] Control node {} was removed or forced down, reconnecting to a different node", (Object)ControlConnection.this.logPrefix, (Object)event.node);
                this.reconnectNow();
            }
        }

        private void forceClose() {
            assert (ControlConnection.this.adminExecutor.inEventLoop());
            if (this.closeWasCalled) {
                return;
            }
            this.closeWasCalled = true;
            LOG.debug("[{}] Starting shutdown", (Object)ControlConnection.this.logPrefix);
            this.reconnection.stop();
            if (ControlConnection.this.channel == null) {
                LOG.debug("[{}] Shutdown complete", (Object)ControlConnection.this.logPrefix);
                this.closeFuture.complete(null);
            } else {
                ControlConnection.this.channel.forceClose().addListener(f -> {
                    if (f.isSuccess()) {
                        LOG.debug("[{}] Shutdown complete", (Object)ControlConnection.this.logPrefix);
                        this.closeFuture.complete(null);
                    } else {
                        this.closeFuture.completeExceptionally(f.cause());
                    }
                });
            }
        }
    }
}

