/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.hollow.api.consumer.fs;

import com.netflix.hollow.api.consumer.HollowConsumer;
import com.netflix.hollow.core.read.OptionalBlobPartInput;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.logging.Logger;

public class HollowFilesystemBlobRetriever
implements HollowConsumer.BlobRetriever {
    private static final Logger LOG = Logger.getLogger(HollowFilesystemBlobRetriever.class.getName());
    private final Path blobStorePath;
    private final HollowConsumer.BlobRetriever fallbackBlobRetriever;
    private final boolean useExistingStaleSnapshot;
    private final Set<String> optionalBlobParts;
    private Duration existingBlobMaxAge = Duration.ofSeconds(-1L);

    public HollowFilesystemBlobRetriever(Path blobStorePath) {
        this(blobStorePath, null, false);
    }

    public HollowFilesystemBlobRetriever(Path blobStorePath, HollowConsumer.BlobRetriever fallbackBlobRetriever) {
        this(blobStorePath, fallbackBlobRetriever, false);
    }

    public HollowFilesystemBlobRetriever(Path blobStorePath, HollowConsumer.BlobRetriever fallbackBlobRetriever, boolean useExistingStaleSnapshot) {
        this(blobStorePath, fallbackBlobRetriever, useExistingStaleSnapshot, Duration.ofSeconds(-1L));
        this.ensurePathExists(blobStorePath);
    }

    public HollowFilesystemBlobRetriever(Path blobStorePath, HollowConsumer.BlobRetriever fallbackBlobRetriever, boolean useExistingStaleSnapshot, Duration existingBlobMaxAge) {
        this.blobStorePath = blobStorePath;
        this.fallbackBlobRetriever = fallbackBlobRetriever;
        this.useExistingStaleSnapshot = useExistingStaleSnapshot;
        this.optionalBlobParts = fallbackBlobRetriever == null ? null : fallbackBlobRetriever.configuredOptionalBlobParts();
        this.existingBlobMaxAge = existingBlobMaxAge;
        this.ensurePathExists(blobStorePath);
    }

    public HollowFilesystemBlobRetriever(Path blobStorePath, Set<String> optionalBlobParts) {
        this.blobStorePath = blobStorePath;
        this.optionalBlobParts = optionalBlobParts;
        this.useExistingStaleSnapshot = true;
        this.fallbackBlobRetriever = null;
        this.ensurePathExists(blobStorePath);
    }

    private void ensurePathExists(Path blobStorePath) {
        try {
            if (!Files.exists(this.blobStorePath, new LinkOption[0])) {
                Files.createDirectories(this.blobStorePath, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Could not create folder for blobRetriever; path=" + blobStorePath, e);
        }
    }

    @Override
    public HollowConsumer.HeaderBlob retrieveHeaderBlob(long desiredVersion) {
        HollowConsumer.HeaderBlob remoteBlob;
        this.cleanupOldBlobs("header-");
        Path exactPath = this.blobStorePath.resolve("header-" + desiredVersion);
        if (Files.exists(exactPath, new LinkOption[0])) {
            return new FilesystemHeaderBlob(exactPath, desiredVersion);
        }
        long maxVersionBeforeDesired = Long.MIN_VALUE;
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(this.blobStorePath);){
            for (Path path : directoryStream) {
                long version;
                String filename = path.getFileName().toString();
                if (!filename.startsWith("header-") || (version = Long.parseLong(filename.substring(filename.lastIndexOf("-") + 1))) >= desiredVersion || version <= maxVersionBeforeDesired) continue;
                maxVersionBeforeDesired = version;
            }
        }
        catch (IOException ex) {
            throw new RuntimeException("Error listing header files; path=" + this.blobStorePath, ex);
        }
        FilesystemHeaderBlob filesystemBlob = null;
        if (maxVersionBeforeDesired != Long.MIN_VALUE) {
            filesystemBlob = new FilesystemHeaderBlob(this.blobStorePath.resolve("snapshot-" + maxVersionBeforeDesired), maxVersionBeforeDesired);
            if (this.useExistingStaleSnapshot) {
                return filesystemBlob;
            }
        }
        if (this.fallbackBlobRetriever != null && (remoteBlob = this.fallbackBlobRetriever.retrieveHeaderBlob(desiredVersion)) != null && (filesystemBlob == null || remoteBlob.getVersion() != filesystemBlob.getVersion())) {
            return new HeaderBlobFromBackupToFilesystem(remoteBlob, this.blobStorePath.resolve("header-" + remoteBlob.getVersion()));
        }
        return filesystemBlob;
    }

    @Override
    public HollowConsumer.Blob retrieveSnapshotBlob(long desiredVersion) {
        HollowConsumer.Blob remoteBlob;
        this.cleanupOldBlobs("snapshot-");
        Path exactPath = this.blobStorePath.resolve("snapshot-" + desiredVersion);
        if (Files.exists(exactPath, new LinkOption[0]) && this.allRequestedPartsExist(HollowConsumer.Blob.BlobType.SNAPSHOT, -1L, desiredVersion)) {
            return this.filesystemBlob(HollowConsumer.Blob.BlobType.SNAPSHOT, -1L, desiredVersion);
        }
        long maxVersionBeforeDesired = Long.MIN_VALUE;
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(this.blobStorePath);){
            for (Path path : directoryStream) {
                long version;
                String filename = path.getFileName().toString();
                if (!filename.startsWith("snapshot-")) continue;
                try {
                    version = Long.parseLong(filename.substring(filename.lastIndexOf("-") + 1));
                }
                catch (NumberFormatException ex) {
                    LOG.info("Ignoring ineligible file in local blob store: " + path);
                    continue;
                }
                if (version >= desiredVersion || version <= maxVersionBeforeDesired || !this.allRequestedPartsExist(HollowConsumer.Blob.BlobType.SNAPSHOT, -1L, version)) continue;
                maxVersionBeforeDesired = version;
            }
        }
        catch (IOException ex) {
            throw new RuntimeException("Error listing snapshot files; path=" + this.blobStorePath, ex);
        }
        HollowConsumer.Blob filesystemBlob = null;
        if (maxVersionBeforeDesired != Long.MIN_VALUE) {
            filesystemBlob = this.filesystemBlob(HollowConsumer.Blob.BlobType.SNAPSHOT, -1L, maxVersionBeforeDesired);
            if (this.useExistingStaleSnapshot) {
                return filesystemBlob;
            }
        }
        if (this.fallbackBlobRetriever != null && (remoteBlob = this.fallbackBlobRetriever.retrieveSnapshotBlob(desiredVersion)) != null && (filesystemBlob == null || remoteBlob.getToVersion() != filesystemBlob.getToVersion())) {
            return new BlobForBackupToFilesystem(remoteBlob, this.blobStorePath.resolve("snapshot-" + remoteBlob.getToVersion()));
        }
        return filesystemBlob;
    }

    private HollowConsumer.Blob filesystemBlob(HollowConsumer.Blob.BlobType type, long currentVersion, long destinationVersion) {
        HashMap<String, Path> optionalPartPaths = null;
        switch (type) {
            case SNAPSHOT: {
                Path path = this.blobStorePath.resolve("snapshot-" + destinationVersion);
                if (this.optionalBlobParts != null && !this.optionalBlobParts.isEmpty()) {
                    optionalPartPaths = new HashMap<String, Path>(this.optionalBlobParts.size());
                    for (String part : this.optionalBlobParts) {
                        optionalPartPaths.put(part, this.blobStorePath.resolve("snapshot_" + part + "-" + destinationVersion));
                    }
                }
                return new FilesystemBlob(path, destinationVersion, optionalPartPaths);
            }
            case DELTA: {
                Path path = this.blobStorePath.resolve("delta-" + currentVersion + "-" + destinationVersion);
                if (this.optionalBlobParts != null && !this.optionalBlobParts.isEmpty()) {
                    optionalPartPaths = new HashMap(this.optionalBlobParts.size());
                    for (String part : this.optionalBlobParts) {
                        optionalPartPaths.put(part, this.blobStorePath.resolve("delta_" + part + "-" + currentVersion + "-" + destinationVersion));
                    }
                }
                return new FilesystemBlob(path, currentVersion, destinationVersion, optionalPartPaths);
            }
            case REVERSE_DELTA: {
                Path path = this.blobStorePath.resolve("reversedelta-" + currentVersion + "-" + destinationVersion);
                if (this.optionalBlobParts != null && !this.optionalBlobParts.isEmpty()) {
                    optionalPartPaths = new HashMap(this.optionalBlobParts.size());
                    for (String part : this.optionalBlobParts) {
                        optionalPartPaths.put(part, this.blobStorePath.resolve("reversedelta_" + part + "-" + currentVersion + "-" + destinationVersion));
                    }
                }
                return new FilesystemBlob(path, currentVersion, destinationVersion, optionalPartPaths);
            }
        }
        throw new IllegalArgumentException("Unknown BlobType: " + type.toString());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public HollowConsumer.Blob retrieveDeltaBlob(long currentVersion) {
        this.cleanupOldBlobs("delta-");
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(this.blobStorePath);){
            for (Path path : directoryStream) {
                long destinationVersion;
                String filename = path.getFileName().toString();
                if (!filename.startsWith("delta-" + currentVersion) || !this.allRequestedPartsExist(HollowConsumer.Blob.BlobType.DELTA, currentVersion, destinationVersion = Long.parseLong(filename.substring(filename.lastIndexOf("-") + 1)))) continue;
                HollowConsumer.Blob blob = this.filesystemBlob(HollowConsumer.Blob.BlobType.DELTA, currentVersion, destinationVersion);
                return blob;
            }
        }
        catch (IOException ex) {
            throw new RuntimeException("Error listing delta files; path=" + this.blobStorePath, ex);
        }
        if (this.fallbackBlobRetriever == null) return null;
        HollowConsumer.Blob remoteBlob = this.fallbackBlobRetriever.retrieveDeltaBlob(currentVersion);
        if (remoteBlob == null) return null;
        return new BlobForBackupToFilesystem(remoteBlob, this.blobStorePath.resolve("delta-" + remoteBlob.getFromVersion() + "-" + remoteBlob.getToVersion()));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public HollowConsumer.Blob retrieveReverseDeltaBlob(long currentVersion) {
        this.cleanupOldBlobs("reversedelta-");
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(this.blobStorePath);){
            for (Path path : directoryStream) {
                long destinationVersion;
                String filename = path.getFileName().toString();
                if (!filename.startsWith("reversedelta-" + currentVersion) || !this.allRequestedPartsExist(HollowConsumer.Blob.BlobType.REVERSE_DELTA, currentVersion, destinationVersion = Long.parseLong(filename.substring(filename.lastIndexOf("-") + 1)))) continue;
                HollowConsumer.Blob blob = this.filesystemBlob(HollowConsumer.Blob.BlobType.REVERSE_DELTA, currentVersion, destinationVersion);
                return blob;
            }
        }
        catch (IOException ex) {
            throw new RuntimeException("Error listing reverse delta files; path=" + this.blobStorePath, ex);
        }
        if (this.fallbackBlobRetriever == null) return null;
        HollowConsumer.Blob remoteBlob = this.fallbackBlobRetriever.retrieveReverseDeltaBlob(currentVersion);
        if (remoteBlob == null) return null;
        return new BlobForBackupToFilesystem(remoteBlob, this.blobStorePath.resolve("reversedelta-" + remoteBlob.getFromVersion() + "-" + remoteBlob.getToVersion()));
    }

    private boolean allRequestedPartsExist(HollowConsumer.Blob.BlobType type, long currentVersion, long destinationVersion) {
        if (this.optionalBlobParts == null || this.optionalBlobParts.isEmpty()) {
            return true;
        }
        for (String part : this.optionalBlobParts) {
            String filename = null;
            switch (type) {
                case SNAPSHOT: {
                    filename = "snapshot_" + part + "-" + destinationVersion;
                    break;
                }
                case DELTA: {
                    filename = "delta_" + part + "-" + currentVersion + "-" + destinationVersion;
                    break;
                }
                case REVERSE_DELTA: {
                    filename = "reversedelta_" + part + "-" + currentVersion + "-" + destinationVersion;
                }
            }
            if (Files.exists(this.blobStorePath.resolve(filename), new LinkOption[0])) continue;
            return false;
        }
        return true;
    }

    private void cleanupOldBlobs(String prefix) {
        if (this.existingBlobMaxAge == null || this.existingBlobMaxAge.isNegative()) {
            return;
        }
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(this.blobStorePath);){
            for (Path blobPath : directoryStream) {
                String filename = blobPath.getFileName().toString();
                if (!filename.startsWith(prefix)) continue;
                long lastModifiedTime = Files.getLastModifiedTime(blobPath, new LinkOption[0]).toMillis();
                long currentTime = System.currentTimeMillis();
                if (currentTime - lastModifiedTime <= this.existingBlobMaxAge.toMillis()) continue;
                Files.delete(blobPath);
                LOG.info("Deleted old blob: " + blobPath + " last modified at " + lastModifiedTime + " max age " + this.existingBlobMaxAge);
            }
        }
        catch (IOException ex) {
            throw new RuntimeException("Error cleaning up old blobs; path=" + this.blobStorePath, ex);
        }
    }

    private static class BlobForBackupToFilesystem
    extends HollowConsumer.Blob {
        private final HollowConsumer.Blob remoteBlob;
        private final Path path;

        BlobForBackupToFilesystem(HollowConsumer.Blob remoteBlob, Path destinationPath) {
            super(remoteBlob.getFromVersion(), remoteBlob.getToVersion());
            this.path = destinationPath;
            this.remoteBlob = remoteBlob;
        }

        @Override
        public InputStream getInputStream() throws IOException {
            Path tempPath = this.path.resolveSibling(this.path.getName(this.path.getNameCount() - 1) + "-" + UUID.randomUUID().toString());
            try (InputStream is = this.remoteBlob.getInputStream();
                 OutputStream os = Files.newOutputStream(tempPath, new OpenOption[0]);){
                int n;
                byte[] buf = new byte[4096];
                while (-1 != (n = is.read(buf))) {
                    os.write(buf, 0, n);
                }
            }
            Files.move(tempPath, this.path, StandardCopyOption.REPLACE_EXISTING);
            return new BufferedInputStream(Files.newInputStream(this.path, new OpenOption[0]));
        }

        @Override
        public File getFile() throws IOException {
            Path tempPath = this.path.resolveSibling(this.path.getName(this.path.getNameCount() - 1) + "-" + UUID.randomUUID().toString());
            try (InputStream is = this.remoteBlob.getInputStream();
                 OutputStream os = Files.newOutputStream(tempPath, new OpenOption[0]);){
                int n;
                byte[] buf = new byte[4096];
                while (-1 != (n = is.read(buf))) {
                    os.write(buf, 0, n);
                }
            }
            Files.move(tempPath, this.path, StandardCopyOption.REPLACE_EXISTING);
            return this.path.toFile();
        }

        @Override
        public OptionalBlobPartInput getOptionalBlobPartInputs() throws IOException {
            OptionalBlobPartInput remoteOptionalParts = this.remoteBlob.getOptionalBlobPartInputs();
            if (remoteOptionalParts == null) {
                return null;
            }
            OptionalBlobPartInput localOptionalParts = new OptionalBlobPartInput();
            for (Map.Entry<String, InputStream> entry : remoteOptionalParts.getInputStreamsByPartName().entrySet()) {
                Path tempPath = this.path.resolveSibling(this.path.getName(this.path.getNameCount() - 1) + "_" + entry.getKey() + "-" + UUID.randomUUID().toString());
                Path destPath = this.getBlobType() == HollowConsumer.Blob.BlobType.SNAPSHOT ? this.path.resolveSibling(this.getBlobType().getType() + "_" + entry.getKey() + "-" + this.getToVersion()) : this.path.resolveSibling(this.getBlobType().getType() + "_" + entry.getKey() + "-" + this.getFromVersion() + "-" + this.getToVersion());
                try (InputStream is = entry.getValue();
                     OutputStream os = Files.newOutputStream(tempPath, new OpenOption[0]);){
                    int n;
                    byte[] buf = new byte[4096];
                    while (-1 != (n = is.read(buf, 0, buf.length))) {
                        os.write(buf, 0, n);
                    }
                }
                Files.move(tempPath, destPath, StandardCopyOption.REPLACE_EXISTING);
                localOptionalParts.addInput(entry.getKey(), new BufferedInputStream(Files.newInputStream(destPath, new OpenOption[0])));
            }
            return localOptionalParts;
        }
    }

    private static class HeaderBlobFromBackupToFilesystem
    extends HollowConsumer.HeaderBlob {
        private final HollowConsumer.HeaderBlob remoteHeaderBlob;
        private final Path path;

        protected HeaderBlobFromBackupToFilesystem(HollowConsumer.HeaderBlob remoteHeaderBlob, Path destinationPath) {
            super(remoteHeaderBlob.getVersion());
            this.path = destinationPath;
            this.remoteHeaderBlob = remoteHeaderBlob;
        }

        @Override
        public InputStream getInputStream() throws IOException {
            Path tempPath = this.path.resolveSibling(this.path.getName(this.path.getNameCount() - 1) + "-" + UUID.randomUUID().toString());
            try (InputStream is = this.remoteHeaderBlob.getInputStream();
                 OutputStream os = Files.newOutputStream(tempPath, new OpenOption[0]);){
                int n;
                byte[] buf = new byte[4096];
                while (-1 != (n = is.read(buf))) {
                    os.write(buf, 0, n);
                }
            }
            Files.move(tempPath, this.path, StandardCopyOption.REPLACE_EXISTING);
            return new BufferedInputStream(Files.newInputStream(this.path, new OpenOption[0]));
        }

        @Override
        public File getFile() throws IOException {
            Path tempPath = this.path.resolveSibling(this.path.getName(this.path.getNameCount() - 1) + "-" + UUID.randomUUID().toString());
            try (InputStream is = this.remoteHeaderBlob.getInputStream();
                 OutputStream os = Files.newOutputStream(tempPath, new OpenOption[0]);){
                int n;
                byte[] buf = new byte[4096];
                while (-1 != (n = is.read(buf))) {
                    os.write(buf, 0, n);
                }
            }
            Files.move(tempPath, this.path, StandardCopyOption.REPLACE_EXISTING);
            return this.path.toFile();
        }
    }

    private static class FilesystemBlob
    extends HollowConsumer.Blob {
        private final Path path;
        private final Map<String, Path> optionalPartPaths;

        @Deprecated
        FilesystemBlob(File snapshotFile, long toVersion) {
            this(snapshotFile.toPath(), toVersion);
        }

        FilesystemBlob(Path snapshotPath, long toVersion) {
            this(snapshotPath, toVersion, null);
        }

        FilesystemBlob(Path deltaPath, long fromVersion, long toVersion) {
            this(deltaPath, fromVersion, toVersion, null);
        }

        FilesystemBlob(Path snapshotPath, long toVersion, Map<String, Path> optionalPartPaths) {
            super(toVersion);
            this.path = snapshotPath;
            this.optionalPartPaths = optionalPartPaths;
        }

        FilesystemBlob(Path deltaPath, long fromVersion, long toVersion, Map<String, Path> optionalPartPaths) {
            super(fromVersion, toVersion);
            this.path = deltaPath;
            this.optionalPartPaths = optionalPartPaths;
        }

        @Override
        public InputStream getInputStream() throws IOException {
            return new BufferedInputStream(Files.newInputStream(this.path, new OpenOption[0]));
        }

        @Override
        public OptionalBlobPartInput getOptionalBlobPartInputs() throws IOException {
            if (this.optionalPartPaths == null || this.optionalPartPaths.isEmpty()) {
                return null;
            }
            OptionalBlobPartInput input = new OptionalBlobPartInput();
            for (Map.Entry<String, Path> pathEntry : this.optionalPartPaths.entrySet()) {
                input.addInput(pathEntry.getKey(), pathEntry.getValue().toFile());
            }
            return input;
        }

        @Override
        public File getFile() throws IOException {
            return this.path.toFile();
        }
    }

    private static class FilesystemHeaderBlob
    extends HollowConsumer.HeaderBlob {
        private final Path path;

        protected FilesystemHeaderBlob(Path headerPath, long version) {
            super(version);
            this.path = headerPath;
        }

        @Override
        public InputStream getInputStream() throws IOException {
            return new BufferedInputStream(Files.newInputStream(this.path, new OpenOption[0]));
        }

        @Override
        public File getFile() throws IOException {
            return this.path.toFile();
        }
    }
}

