/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.cql.store;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BoundStatement;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.shaded.guava.common.collect.Lists;

import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.cql.CqlFailedConstraintException;
import net.bluemind.cql.CqlPersistenceException;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.repository.sequences.ISequenceStore;
import net.bluemind.repository.sequences.Sequences;

public class CqlContainerStore extends CqlAbstractStore implements IContainerStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlContainerStore.class);
	private final SecurityContext ctx;
	private final ISequenceStore seqs;

	public CqlContainerStore(CqlSession s, ISequenceStore seq, SecurityContext sec) {
		super(s);
		this.seqs = seq;
		this.ctx = sec;
	}

	private List<String> owned(String owner) {
		return map("""
				SELECT
				uid
				FROM
				t_container_by_owner
				WHERE
				owner=?
				""", r -> r.getString(0), voidPop(), owner);
	}

	private List<String> uidByType(String type) {
		return map("""
				SELECT
				uid
				FROM
				t_container_by_type
				WHERE
				type=?
				""", r -> r.getString(0), voidPop(), type);
	}

	@Override
	public List<Container> findByContainerQuery(ContainerQuery containerQuery) throws SQLException {
		List<String> owned = owned(containerQuery.owner);
		List<Container> mgetOwnedByType = mget(owned).stream().filter(c -> c.type.equals(containerQuery.type)).toList();
		if (containerQuery.name != null) {
			return mgetOwnedByType.stream().filter(c -> c.name.equals(containerQuery.name)).toList();
		}
		return mgetOwnedByType;
	}

	@Override
	public List<Container> findByType(String containerType) throws SQLException {
		return mget(uidByType(containerType));
	}

	@Override
	public List<Container> findAccessiblesByType(ContainerQuery query) throws SQLException {
		List<String> myUids = map("select uid from t_container_by_owner where owner=?", r -> r.getString(0), voidPop(),
				ctx.getSubject());

		Set<String> subjects = new HashSet<>();
		subjects.add(ctx.getSubject());
		subjects.add(ctx.getContainerUid());
		subjects.addAll(ctx.getMemberOf());

		Set<Long> sharedToMeIds = new HashSet<>();
		for (List<String> subSlice : Lists.partition(List.copyOf(subjects), 64)) {
			List<Long> shared = map("select container_id from t_container_acl where subject IN ?", r -> r.getLong(0),
					voidPop(), subSlice);
			sharedToMeIds.addAll(shared);
		}
		List<String> sharedUids = mgetUids(List.copyOf(sharedToMeIds));
		List<String> merged = Stream.concat(myUids.stream(), sharedUids.stream()).toList();
		List<Container> unfiltered = mget(merged);
		if (query.type != null) {
			unfiltered = unfiltered.stream().filter(c -> c.type.equals(query.type)).toList();
		}
		if (query.name != null) {
			unfiltered = unfiltered.stream().filter(c -> c.name.contains(query.name)).toList();
		}
		logger.info("accessible({}) -> {} container(s)", query, unfiltered.size());
		return unfiltered;
	}

	@Override
	public Container create(Container c) throws SQLException {
		PreparedStatement prep = session.prepare("""
				INSERT INTO t_container
				(id, owner, domain_uid, uid, name, type, createdby, updatedby, created, updated, default_container)
				values
				(?, ?, ?, ?, ?, ?, ?, ?, toTimestamp(now()), toTimestamp(now()), ?)
				IF NOT EXISTS
				""");
		c.id = seqs.nextVal(Sequences.ownerIds("containers_allocator"));
		String principal = ctx.getSubject();
		ResultSet result = session.execute(
				prep.bind(c.id, c.owner, c.domainUid, c.uid, c.name, c.type, principal, principal, c.defaultContainer));

		if (!result.wasApplied()) {
			throw new CqlFailedConstraintException("container " + c + " looks like a duplicate");
		}
		logger.info("CQL create container {}", c.uid);

		voidCql("UPDATE cnt_items_unseen SET unseen=unseen+1 WHERE container_id=?", c.id);
		voidCql("UPDATE cnt_items_unseen SET unseen=unseen-1 WHERE container_id=?", c.id);
		voidCql("UPDATE cnt_items_unseen_visible SET unseen=unseen+1 WHERE container_id=?", c.id);
		voidCql("UPDATE cnt_items_unseen_visible SET unseen=unseen-1 WHERE container_id=?", c.id);
		voidCql("UPDATE cnt_items_all SET all=all+1 WHERE container_id=?", c.id);
		voidCql("UPDATE cnt_items_all SET all=all-1 WHERE container_id=?", c.id);

		long curVal = seqs.curVal(Sequences.itemVersions(c.uid));
		logger.info("Init versions sequence for {} -> {}", c.uid, curVal);

		return c;

	}

	private static final EntityPopulator<Container> POP = (Row r, int i, Container c) -> {
		c.id = r.getLong(i++);
		c.owner = r.getString(i++);
		c.domainUid = r.getString(i++);
		c.uid = r.getString(i++);
		c.name = r.getString(i++);
		c.type = r.getString(i++);
		c.createdBy = r.getString(i++);
		c.updatedBy = r.getString(i++);
		c.created = Date.from(r.getInstant(i++));
		c.updated = Date.from(r.getInstant(i++));
		c.defaultContainer = r.getBoolean(i++);
		return i;
	};

	public List<Container> mget(List<String> uid) {
		List<Container> out = new ArrayList<>(uid.size());
		for (var slice : Lists.partition(uid, 64)) {
			out.addAll(map("""
					SELECT
					id, owner, domain_uid, uid, name, type, createdby, updatedby, created, updated, default_container
					FROM
					t_container
					WHERE
					uid IN ?
					""", r -> new Container(), POP, slice));
		}
		return out;
	}

	@Override
	public boolean exists(String uid) throws SQLException {
		return map("select id from t_container where uid = ?", r -> r.getLong(0), voidPop(), uid) != null;
	}

	public List<String> mgetUids(List<Long> ids) {
		List<String> out = new ArrayList<>(ids.size());
		for (var slice : Lists.partition(ids, 64)) {
			out.addAll(map("""
					SELECT uid FROM t_container_by_id
					WHERE id IN ?
					""", r -> r.getString(0), voidPop(), slice));
		}
		return out;
	}

	@Override
	public void update(String uid, String name, boolean defaultContainer) throws SQLException {
		PreparedStatement prep = session.prepare("""
				UPDATE t_container SET
				name = ?,
				default_container = ?,
				updated = toUnixTimestamp(now()),
				updatedby = ?
				WHERE uid = ?
				""");
		ResultSet result = session.execute(prep.bind(name, defaultContainer, ctx.getSubject(), uid));
		if (!result.wasApplied()) {
			throw new CqlPersistenceException("Update to " + uid + " was not applied");
		}
	}

	@Override
	public Container get(String uid) {
		return unique("""
				SELECT
				id, owner, domain_uid, uid, name, type, createdby, updatedby, created, updated, default_container
				FROM
				t_container
				WHERE
				uid = ?
				""", r -> new Container(), POP, uid);
	}

	@Override
	public Container get(long id) throws SQLException {
		String uid = unique("""
				SELECT uid FROM t_container_by_id WHERE id = ?
				""", r -> r.getString(0), voidPop(), id);
		if (uid == null) {
			return null;
		}
		return get(uid);
	}

	@Override
	public void delete(String uid) throws SQLException {
		Container fetched = get(uid);
		if (fetched == null) {
			return;
		}
		delete(fetched);
	}

	@Override
	public void delete(Container existing) throws SQLException {
		voidCql("DELETE FROM t_container WHERE uid = ?", existing.uid);
		seqs.drop(Sequences.itemVersions(existing.uid));
	}

	@Override
	public void deleteAllSubscriptions(Container container) throws SQLException {
		List<String> subjects = listSubscriptions(container);
		BoundStatement[] statements = subjects.stream()
				.map(s -> b("delete from t_subs_by_subject where subject=? and container_id=?", s, container.id))
				.toArray(BoundStatement[]::new);
		batch(statements);
	}

	@Override
	public List<String> listSubscriptions(Container container) throws SQLException {
		return map("select subject from t_subs_by_container where container_id=?", r -> r.getString(0), voidPop(),
				container.id);
	}

}
