package net.bluemind.cql;

import java.net.InetSocketAddress;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.AllNodesFailedException;
import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.NoNodeAvailableException;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverConfigLoader;

import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.IServer;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class CqlSessions {

	private static final Logger logger = LoggerFactory.getLogger(CqlSessions.class);
	private static Map<String, CqlSession> byKeyspace = new ConcurrentHashMap<>();

	public static final boolean UNSAFE_FOR_TESTS = Boolean.getBoolean("cql.unsafe.for.tests");

	private CqlSessions() {

	}

	private static CqlSession sysSession(ItemValue<Server> srv) {
		// speedup schema creation in tests
		var configBuild = DriverConfigLoader.programmaticBuilder();
		configBuild = configBuild.withDuration(DefaultDriverOption.REQUEST_TIMEOUT, reqTimeout());
		if (UNSAFE_FOR_TESTS) {
			logger.info("Unsafe for tests settings actived {}", UNSAFE_FOR_TESTS);
			configBuild = configBuild.withInt(DefaultDriverOption.METADATA_SCHEMA_MAX_EVENTS, 1);
		}
		return CqlSession.builder()//
				.addContactPoint(InetSocketAddress.createUnresolved(srv.value.address(), 9042))//
				.withConfigLoader(configBuild.build())//
				.withLocalDatacenter("datacenter1")//
				.build();
	}

	private static CqlSession retryingSysSession(ItemValue<Server> srv) {
		int attempts = 0;
		AllNodesFailedException lastErr = null;
		do {
			if (lastErr != null) {
				try {
					Thread.sleep(250);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
				logger.info("System connection to {} attempt {}...", srv.value, attempts);
			}
			try {
				return sysSession(srv);
			} catch (AllNodesFailedException nfe) {
				logger.warn("Retrying connection to {} ({})", srv.value, nfe.getMessage());
				lastErr = nfe;
			}
		} while (++attempts < 40);
		throw lastErr;
	}

	private static Duration reqTimeout() {
		return CoreConfig.get().getDuration(CoreConfig.Cassandra.REQUEST_TIMEOUT);
	}

	private static CqlSession ksSession(ItemValue<Server> srv, String ks) {
		// speedup schema creation in tests
		var configBuild = DriverConfigLoader.programmaticBuilder();

		configBuild = configBuild.withDuration(DefaultDriverOption.REQUEST_TIMEOUT, reqTimeout());
		if (UNSAFE_FOR_TESTS) {
			logger.info("Unsafe for tests settings actived {}", UNSAFE_FOR_TESTS);
			configBuild = configBuild.withInt(DefaultDriverOption.METADATA_SCHEMA_MAX_EVENTS, 1);
		}
		return CqlSession.builder()//
				.addContactPoint(InetSocketAddress.createUnresolved(srv.value.address(), 9042))//
				.withKeyspace(ks)//
				.withConfigLoader(configBuild.build())//
				.withLocalDatacenter("datacenter1")//
				.build();
	}

	public static CqlSession system(IServer api) {
		return api.allComplete().stream().filter(i -> i.value.tags.contains(TagDescriptor.cql_node.getTag())).findAny()
				.map(CqlSessions::retryingSysSession).orElseThrow(NoNodeAvailableException::new);

	}

	public static CqlSession system() {
		return Topology.getIfAvailable().map(topo -> topo.any(TagDescriptor.cql_node.getTag()))
				.map(CqlSessions::retryingSysSession).orElseThrow(NoNodeAvailableException::new);
	}

	public static CqlSession forKeyspace(String ksId, IServer api) {
		return byKeyspace.computeIfAbsent(ksId,
				ks -> api.allComplete().stream().filter(i -> i.value.tags.contains(TagDescriptor.cql_node.getTag()))
						.findAny().map(srv -> ksSession(srv, ksId)).orElseThrow(NoNodeAvailableException::new));
	}

	public static CqlSession forKeyspace(String ksId) {
		return byKeyspace.computeIfAbsent(ksId,
				ks -> Topology.getIfAvailable().map(topo -> topo.any(TagDescriptor.cql_node.getTag()))
						.map(srv -> ksSession(srv, ksId)).orElseThrow(NoNodeAvailableException::new));
	}

}
